"""
Modelos para el seguimiento de salud del sistema y métricas de performance.
Soporte para modo desarrollador y notificaciones inteligentes.
"""

import uuid

from sqlalchemy import JSON, Boolean, Column, DateTime, Float, Integer, String, Text
from sqlalchemy.dialects.postgresql import UUID

from app.database import Base
from app.utils.datetime_utils import utc_now


class SystemHealthMetric(Base):
    """
    Métricas de salud del sistema para farmacéuticos y desarrolladores.
    """

    __tablename__ = "system_health_metrics"
    __table_args__ = {"extend_existing": True}

    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Categoría de métrica
    category = Column(String(50), nullable=False)  # 'sync', 'performance', 'database', 'api'
    metric_name = Column(String(100), nullable=False)  # 'cima_sync_duration', 'memory_usage'
    metric_value = Column(Float, nullable=False)
    metric_unit = Column(String(20))  # 'seconds', 'MB', 'percentage', 'count'

    # Contexto farmacéutico
    context = Column(JSON)  # {'pharmacy_id': 'xxx', 'process': 'catalog_sync', 'chunk': 5}

    # Timestamps
    measured_at = Column(DateTime, default=utc_now, nullable=False)
    created_at = Column(DateTime, default=utc_now, nullable=False)


class SystemAlert(Base):
    """
    Alertas inteligentes para administradores de farmacia.
    """

    __tablename__ = "system_alerts"
    __table_args__ = {"extend_existing": True}

    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Clasificación de alerta
    alert_type = Column(String(50), nullable=False)  # 'cima_outdated', 'memory_warning', 'sync_failure'
    severity = Column(String(20), nullable=False)  # 'low', 'medium', 'high', 'critical'
    title = Column(String(200), nullable=False)
    message = Column(Text, nullable=False)

    # Contexto específico
    context = Column(JSON)  # Datos específicos del problema

    # Estado
    is_active = Column(Boolean, default=True, nullable=False)
    is_acknowledged = Column(Boolean, default=False, nullable=False)
    acknowledged_at = Column(DateTime)
    acknowledged_by = Column(String(100))  # username o 'system'

    # Recurrencia
    first_occurred = Column(DateTime, default=utc_now, nullable=False)
    last_occurred = Column(DateTime, default=utc_now, nullable=False)
    occurrence_count = Column(Integer, default=1, nullable=False)

    # Timestamps
    created_at = Column(DateTime, default=utc_now, nullable=False)
    updated_at = Column(DateTime, default=utc_now, onupdate=utc_now, nullable=False)


class DeveloperLog(Base):
    """
    Logs técnicos detallados para modo desarrollador.
    """

    __tablename__ = "developer_logs"
    __table_args__ = {"extend_existing": True}

    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Clasificación
    level = Column(String(20), nullable=False)  # 'DEBUG', 'INFO', 'WARNING', 'ERROR'
    logger_name = Column(String(100), nullable=False)  # 'catalog_service', 'cima_api', 'enrichment'

    # Contenido
    message = Column(Text, nullable=False)
    stack_trace = Column(Text)  # Solo para errores

    # Contexto farmacéutico
    context = Column(JSON)  # {'process': 'sync', 'pharmacy_id': 'xxx', 'file_id': 'yyy'}

    # Performance data
    execution_time_ms = Column(Float)
    memory_usage_mb = Column(Float)

    # Timestamps
    timestamp = Column(DateTime, default=utc_now, nullable=False)
    created_at = Column(DateTime, default=utc_now, nullable=False)


class PerformanceSnapshot(Base):
    """
    Snapshots de performance del sistema para análisis de trends.
    """

    __tablename__ = "performance_snapshots"
    __table_args__ = {"extend_existing": True}

    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Sistema general
    cpu_usage_percent = Column(Float)
    memory_usage_mb = Column(Float)
    memory_available_mb = Column(Float)

    # Base de datos
    db_connections_active = Column(Integer)
    db_connections_idle = Column(Integer)
    db_queries_per_second = Column(Float)
    db_avg_query_time_ms = Column(Float)

    # Redis
    redis_memory_usage_mb = Column(Float)
    redis_keys_count = Column(Integer)
    redis_operations_per_second = Column(Float)

    # APIs externas
    cima_api_response_time_ms = Column(Float)
    nomenclator_api_response_time_ms = Column(Float)
    api_errors_last_hour = Column(Integer)

    # Catálogo
    catalog_total_products = Column(Integer)
    catalog_with_cima = Column(Integer)
    catalog_with_nomenclator = Column(Integer)
    catalog_freshness_hours = Column(Float)

    # Timestamps
    snapshot_at = Column(DateTime, default=utc_now, nullable=False)
    created_at = Column(DateTime, default=utc_now, nullable=False)
