# backend/app/models/seasonality_index.py
"""
Modelo para gestionar indices de estacionalidad por categoria.

Issue #507: Indice de Estacionalidad por Categoria
Permite ajustar metricas de momentum por expectativa estacional,
evitando falsos positivos en Decision Box (ej: protector solar -45% en octubre = normal).

La estacionalidad se aplica SOLO a metricas de corto plazo (momentum),
NO a MAT (Moving Annual Total) que ya tiene estacionalidad "cancelada".
"""

import uuid

from sqlalchemy import (
    CheckConstraint,
    Column,
    DateTime,
    ForeignKey,
    Integer,
    Numeric,
    String,
    UniqueConstraint,
)
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from ..database import Base
from ..utils.datetime_utils import utc_now


# Datos sector default para 7 categorias estacionales
# Formato: {category: [ene, feb, mar, abr, may, jun, jul, ago, sep, oct, nov, dic]}
SECTOR_SEASONALITY_DEFAULTS = {
    "proteccion_solar": [0.3, 0.4, 0.6, 0.8, 1.2, 1.5, 1.8, 1.6, 1.0, 0.5, 0.3, 0.3],
    "antigripales": [1.5, 1.3, 1.0, 0.7, 0.5, 0.4, 0.3, 0.4, 0.6, 1.0, 1.4, 1.6],
    "alergias": [0.5, 0.6, 1.2, 1.6, 1.5, 1.2, 0.8, 0.6, 0.7, 0.6, 0.5, 0.5],
    "adelgazantes": [1.4, 1.3, 1.2, 1.1, 1.0, 0.9, 0.8, 0.8, 1.1, 1.0, 0.9, 0.7],
    "dermocosmetica": [0.9, 0.9, 1.0, 1.0, 1.1, 1.2, 1.3, 1.2, 1.0, 0.9, 0.8, 0.9],
    "repelentes_picaduras": [0.2, 0.3, 0.5, 0.8, 1.2, 1.5, 1.8, 1.5, 0.8, 0.4, 0.2, 0.2],
    "hidratacion": [1.2, 1.1, 1.0, 0.9, 0.8, 0.9, 1.0, 1.0, 0.9, 1.0, 1.1, 1.2],
}


class SeasonalityIndex(Base):
    """
    Modelo para indices de estacionalidad por categoria y mes.

    Permite ajustar expectativas de venta segun temporada:
    - index_value = 1.0: mes promedio (sin ajuste)
    - index_value < 1.0: temporada baja (ej: solar en invierno)
    - index_value > 1.0: temporada alta (ej: antigripal en invierno)

    Prioridad de indices:
    1. Indice especifico de farmacia (pharmacy_id != NULL)
    2. Indice sector (pharmacy_id = NULL)
    3. Default 1.0 (sin ajuste)
    """

    __tablename__ = "seasonality_indices"

    # Clave primaria
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Categoria NECESIDAD L1
    category = Column(
        String(50),
        nullable=False,
        index=True,
        comment="Categoria NECESIDAD L1 (proteccion_solar, antigripales, etc.)",
    )

    # Mes (1-12)
    month = Column(
        Integer,
        nullable=False,
        index=True,
        comment="Mes del anio (1=enero, 12=diciembre)",
    )

    # Valor del indice estacional
    # Numeric(4, 2) para headroom futuro (soporta hasta 99.99)
    index_value = Column(
        Numeric(4, 2),
        nullable=False,
        default=1.0,
        comment="Indice estacional: 1.0=promedio, 0.5=baja, 1.5=alta",
    )

    # Farmacia (NULL = default sector)
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=True,
        index=True,
        comment="ID farmacia o NULL para defaults de sector",
    )

    # Fuente del indice
    source = Column(
        String(20),
        nullable=False,
        default="sector_default",
        comment="Origen: sector_default, calculated, manual",
    )

    # Metadata
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now)

    # Relaciones
    pharmacy = relationship("Pharmacy", back_populates="seasonality_indices")

    # Constraints
    __table_args__ = (
        # Unicidad: categoria + mes + farmacia
        UniqueConstraint(
            "category",
            "month",
            "pharmacy_id",
            name="uq_seasonality_category_month_pharmacy",
        ),
        # Mes valido (1-12)
        CheckConstraint("month >= 1 AND month <= 12", name="ck_seasonality_month_range"),
        # Indice valido (0.1 - 3.0)
        CheckConstraint(
            "index_value >= 0.1 AND index_value <= 3.0",
            name="ck_seasonality_index_range",
        ),
        {"extend_existing": True},
    )

    def to_dict(self) -> dict:
        """Convierte instancia a diccionario para serializacion."""
        return {
            "id": str(self.id),
            "category": self.category,
            "month": self.month,
            "index_value": float(self.index_value) if self.index_value else 1.0,
            "pharmacy_id": str(self.pharmacy_id) if self.pharmacy_id else None,
            "source": self.source,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def __repr__(self) -> str:
        return (
            f"<SeasonalityIndex(category={self.category}, month={self.month}, "
            f"index={self.index_value}, pharmacy_id={self.pharmacy_id})>"
        )
