# backend/app/models/sales_enrichment.py
"""
Modelo SalesEnrichment - Tabla de relación para datos de ventas enriquecidos
Conecta SalesData con ProductCatalog y almacena metadatos de enriquecimiento
"""

import uuid

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Index, Integer, Numeric, String, Text, UniqueConstraint
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from .base import Base

# Import directo para evitar importaciones dinámicas problemáticas
# Movido al top para prevenir conflictos de SQLAlchemy MetaData


class SalesEnrichment(Base):
    """
    Tabla de enriquecimiento para datos de ventas

    Funcionalidades:
    - Conecta SalesData con ProductCatalog
    - Almacena metadatos del proceso de matching
    - Cachea cálculos de análisis farmacéuticos
    - Tracking de calidad del enriquecimiento
    """

    __tablename__ = "sales_enrichment"

    # Clave primaria
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # === RELACIONES ===
    sales_data_id = Column(UUID(as_uuid=True), ForeignKey("sales_data.id"), nullable=False, index=True)
    product_catalog_id = Column(UUID(as_uuid=True), ForeignKey("product_catalog.id"), nullable=True, index=True)

    # === INFORMACIÓN DEL MATCHING ===
    match_method = Column(String(50), nullable=True)  # 'national_code', 'name_fuzzy', 'ean', 'manual'
    match_confidence = Column(Integer, nullable=True)  # 1-100
    match_score = Column(Numeric(5, 2), nullable=True)  # Score detallado del matching

    # === ESTADO DEL ENRIQUECIMIENTO ===
    enrichment_status = Column(
        String(30), nullable=False, default="pending", index=True
    )  # 'pending', 'enriched', 'failed', 'manual_review'
    enrichment_source = Column(String(50), nullable=True)  # 'CIMA', 'nomenclator', 'CIMAVet', 'ML'
    enrichment_version = Column(String(20), nullable=True)  # Versión del algoritmo usado

    # === DATOS FARMACÉUTICOS ENRIQUECIDOS (CACHE) ===
    therapeutic_category = Column(String(200), nullable=True, index=True)  # Categoría terapéutica derivada de ATC
    prescription_category = Column(String(100), nullable=True, index=True)  # Categoría prescripción farmacéutica
    product_type = Column(String(50), nullable=True, index=True)  # 'prescription', 'venta_libre'

    # === ANÁLISIS DE GENÉRICOS (CACHE) ===
    has_generic_alternative = Column(Boolean, nullable=True, default=False, index=True)
    cheapest_generic_price = Column(Numeric(10, 4), nullable=True)  # Precio del genérico más barato
    potential_generic_savings = Column(Numeric(10, 4), nullable=True)  # Ahorro potencial
    generic_savings_percent = Column(Numeric(5, 2), nullable=True)  # % ahorro

    # === ANÁLISIS FINANCIERO (CACHE) ===
    price_vs_official = Column(Numeric(10, 4), nullable=True)  # Diferencia vs PVP oficial
    margin_analysis = Column(JSONB, nullable=True)  # Análisis de márgenes
    pricing_category = Column(String(50), nullable=True)  # 'below_official', 'at_official', 'above_official'

    # === CLASIFICACIÓN ML (PARA VENTA LIBRE) ===
    ml_category = Column(String(100), nullable=True, index=True)  # Categoría inferida por ML (NECESIDAD nivel 1)
    ml_confidence = Column(Numeric(5, 2), nullable=True)  # Confianza ML 0-1
    ml_model_version = Column(String(20), nullable=True)  # Versión del modelo ML usado

    # === TAXONOMÍA JERÁRQUICA VENTA LIBRE (Issue #446) ===
    # ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md
    ml_subcategory = Column(
        String(100), nullable=True, index=True,
        comment="Subcategoría NECESIDAD nivel 2 (ej: 'facial', 'corporal')"
    )
    detected_brand = Column(
        String(100), nullable=True, index=True,
        comment="Marca detectada del producto (ej: 'ISDIN', 'Heliocare')"
    )
    brand_line = Column(
        String(200), nullable=True,
        comment="Línea de producto dentro de la marca (ej: 'Fusion Water', 'Ultra')"
    )
    intercambiable_group_id = Column(
        UUID(as_uuid=True),
        ForeignKey("intercambiable_group.id"),
        nullable=True,
        index=True,
        comment="FK a grupo de productos intercambiables (nivel 4)"
    )
    product_cluster_id = Column(
        UUID(as_uuid=True),
        ForeignKey("product_cluster.id"),
        nullable=True,
        index=True,
        comment="FK a cluster jerárquico para clasificación automática (Issue #456)"
    )

    # === CATÁLOGO VENTA LIBRE (Issue #457) ===
    venta_libre_product_id = Column(
        UUID(as_uuid=True),
        ForeignKey("product_catalog_venta_libre.id"),
        nullable=True,
        index=True,
        comment="FK a catálogo interno de venta libre (productos sin CN)"
    )

    # === ENRIQUECIMIENTO LLM (Issue #456) ===
    llm_indicaciones = Column(
        Text,
        nullable=True,
        comment="Indicaciones del producto extraídas por LLM"
    )
    llm_composicion = Column(
        Text,
        nullable=True,
        comment="Composición principal extraída por LLM"
    )
    llm_modo_empleo = Column(
        Text,
        nullable=True,
        comment="Modo de empleo extraído por LLM"
    )
    llm_confidence = Column(
        Numeric(3, 2),
        nullable=True,
        comment="Confianza del LLM en la extracción (0.00-1.00)"
    )
    llm_enriched_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Timestamp de enriquecimiento LLM"
    )
    llm_model_version = Column(
        String(50),
        nullable=True,
        comment="Versión del modelo LLM usado (ej: 'qwen2.5:7b-instruct')"
    )

    # === METADATOS DE PROCESAMIENTO ===
    processing_time_ms = Column(Integer, nullable=True)  # Tiempo de procesamiento en ms
    error_messages = Column(Text, nullable=True)  # Mensajes de error si falló
    warning_messages = Column(Text, nullable=True)  # Warnings durante procesamiento
    manual_review_reason = Column(String(200), nullable=True)  # Razón para revisión manual

    # === INFORMACIÓN CONTEXTUAL ===
    seasonal_factor = Column(Numeric(5, 2), nullable=True)  # Factor estacional si aplica
    cross_selling_opportunities = Column(JSONB, nullable=True)  # Oportunidades cross-selling
    stock_analysis = Column(JSONB, nullable=True)  # Análisis de stock si disponible

    # === TIMESTAMPS ===
    enriched_at = Column(DateTime(timezone=True), nullable=True)  # Cuándo se enriqueció
    last_updated = Column(DateTime(timezone=True), server_default=func.now(), onupdate=func.now())
    created_at = Column(DateTime(timezone=True), server_default=func.now(), nullable=False)

    # === RELACIONES SQLALCHEMY ===
    sales_data = relationship("SalesData", back_populates="enrichment", lazy="joined")
    product_catalog = relationship("ProductCatalog", lazy="joined")
    intercambiable_group = relationship(
        "IntercambiableGroup",
        back_populates="sales_enrichments",
        lazy="joined"
    )
    product_cluster = relationship(
        "ProductCluster",
        back_populates="sales_enrichments",
        lazy="joined"
    )
    corrections = relationship(
        "ProductCorrection",
        back_populates="sales_enrichment",
        lazy="selectin",
        cascade="all, delete-orphan"
    )
    venta_libre_product = relationship(
        "ProductCatalogVentaLibre",
        back_populates="sales_enrichments",
        lazy="joined"
    )

    # === ÍNDICES COMPUESTOS ===
    __table_args__ = (
        # Índice único para evitar duplicados sales_data -> enrichment
        UniqueConstraint("sales_data_id", name="uq_sales_enrichment_sales_data"),
        # Índice para queries de análisis
        Index("idx_enrichment_status_confidence", "enrichment_status", "match_confidence"),
        # Índice para análisis de genéricos
        Index(
            "idx_generic_analysis",
            "has_generic_alternative",
            "potential_generic_savings",
        ),
        # Índice para análisis por categorías
        Index("idx_therapeutic_category", "therapeutic_category", "product_type"),
        # Índice para análisis ML
        Index("idx_ml_category_confidence", "ml_category", "ml_confidence"),
        # Índice para búsquedas por farmacia
        Index("idx_enrichment_pharmacy_lookup", "sales_data_id", "enrichment_status"),
        # CRÍTICO: extend_existing protege contra conflictos MetaData en production
        {"extend_existing": True},
    )

    def __repr__(self):
        return f"<SalesEnrichment(id={self.id}, sales_id={self.sales_data_id}, status='{self.enrichment_status}')>"

    def to_dict(self, include_relations: bool = False) -> dict:
        """
        Convierte el objeto a diccionario para serialización

        Args:
            include_relations: Incluir datos de SalesData y ProductCatalog

        Returns:
            Diccionario con datos del enriquecimiento
        """
        base_data = {
            "id": str(self.id),
            "sales_data_id": str(self.sales_data_id),
            "product_catalog_id": (str(self.product_catalog_id) if self.product_catalog_id else None),
            "match_method": self.match_method,
            "match_confidence": self.match_confidence,
            "enrichment_status": self.enrichment_status,
            "enrichment_source": self.enrichment_source,
            "therapeutic_category": self.therapeutic_category,
            "prescription_category": self.prescription_category,
            "product_type": self.product_type,
            "has_generic_alternative": self.has_generic_alternative,
            "potential_generic_savings": (
                float(self.potential_generic_savings) if self.potential_generic_savings else None
            ),
            "generic_savings_percent": (float(self.generic_savings_percent) if self.generic_savings_percent else None),
            "ml_category": self.ml_category,
            "enriched_at": self.enriched_at.isoformat() if self.enriched_at else None,
            "last_updated": (self.last_updated.isoformat() if self.last_updated else None),
        }

        if include_relations:
            if self.sales_data:
                base_data["sales_data"] = self.sales_data.to_dict()
            if self.product_catalog:
                base_data["product_catalog"] = self.product_catalog.to_dict()

        return base_data

    # TEMPORALMENTE DESHABILITADO - PROBLEMA CIRCULAR IMPORTS
    # TODO: Reescribir sin importaciones dinámicas
    @classmethod
    def get_enrichment_stats_DISABLED(cls, session, pharmacy_id: str = None):
        """
        Obtiene estadísticas de enriquecimiento

        Args:
            session: Sesión de SQLAlchemy
            pharmacy_id: ID de farmacia específica (opcional)

        Returns:
            Diccionario con estadísticas de enriquecimiento
        """
        # Importación dinámica para evitar circular imports
        from .sales_data import SalesData

        query = session.query(cls).join(SalesData)

        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        total_records = query.count()

        # Estadísticas por estado
        status_counts = {}
        for status, count in query.group_by(cls.enrichment_status).values(cls.enrichment_status, func.count(cls.id)):
            status_counts[status] = count

        # Estadísticas de confianza
        avg_confidence = (
            query.filter(cls.match_confidence.isnot(None)).with_entities(func.avg(cls.match_confidence)).scalar()
        )

        # Estadísticas de genéricos
        with_generics = query.filter(cls.has_generic_alternative.is_(True)).count()

        total_savings = (
            query.filter(cls.potential_generic_savings.isnot(None))
            .with_entities(func.sum(cls.potential_generic_savings))
            .scalar()
        )

        return {
            "total_records": total_records,
            "status_breakdown": status_counts,
            "average_confidence": float(avg_confidence) if avg_confidence else 0,
            "enrichment_rate": ((status_counts.get("enriched", 0) / total_records * 100) if total_records > 0 else 0),
            "generic_opportunities": with_generics,
            "total_potential_savings": float(total_savings) if total_savings else 0,
        }

    @classmethod
    def get_by_sales_data_id(cls, session, sales_data_id: str):
        """
        Obtiene enriquecimiento por ID de sales_data

        Args:
            session: Sesión de SQLAlchemy
            sales_data_id: ID del registro de venta

        Returns:
            SalesEnrichment o None
        """
        return session.query(cls).filter(cls.sales_data_id == sales_data_id).first()

    # TEMPORALMENTE DESHABILITADO - PROBLEMA CIRCULAR IMPORTS
    # TODO: Reescribir sin importaciones dinámicas
    @classmethod
    def get_generic_opportunities_DISABLED(cls, session, pharmacy_id: str = None, min_savings: float = 0.50):
        """
        Obtiene oportunidades de genéricos con ahorro mínimo

        Args:
            session: Sesión de SQLAlchemy
            pharmacy_id: ID de farmacia específica
            min_savings: Ahorro mínimo en euros

        Returns:
            Lista de oportunidades ordenadas por ahorro potencial
        """
        # Importación dinámica para evitar circular imports
        from .sales_data import SalesData

        query = (
            session.query(cls)
            .join(SalesData)
            .filter(
                cls.has_generic_alternative.is_(True),
                cls.potential_generic_savings >= min_savings,
            )
        )

        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        return query.order_by(cls.potential_generic_savings.desc()).all()

    # TEMPORALMENTE DESHABILITADO - PROBLEMA CIRCULAR IMPORTS
    # TODO: Reescribir sin importaciones dinámicas
    @classmethod
    def get_by_therapeutic_category_DISABLED(cls, session, category: str, pharmacy_id: str = None):
        """
        Obtiene ventas por categoría terapéutica

        Args:
            session: Sesión de SQLAlchemy
            category: Categoría terapéutica
            pharmacy_id: ID de farmacia específica

        Returns:
            Lista de ventas enriquecidas de esa categoría
        """
        # Importación dinámica para evitar circular imports
        from .sales_data import SalesData

        query = session.query(cls).join(SalesData).filter(cls.therapeutic_category.ilike(f"%{category}%"))

        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        return query.all()

    # TEMPORALMENTE DESHABILITADO - PROBLEMA CIRCULAR IMPORTS
    # TODO: Reescribir sin importaciones dinámicas
    @classmethod
    def get_prescription_analysis_DISABLED(cls, session, pharmacy_id: str = None):
        """
        Análisis de productos de prescripción vs venta libre

        Args:
            session: Sesión de SQLAlchemy
            pharmacy_id: ID de farmacia específica

        Returns:
            Diccionario con análisis de prescripción
        """
        # Importación dinámica para evitar circular imports
        from .sales_data import SalesData

        query = session.query(cls).join(SalesData)

        if pharmacy_id:
            query = query.filter(SalesData.pharmacy_id == pharmacy_id)

        # Análisis por tipo de producto
        product_type_analysis = {}
        for product_type, count, total_amount in (
            query.filter(cls.product_type.isnot(None))
            .join(SalesData)
            .group_by(cls.product_type)
            .values(cls.product_type, func.count(cls.id), func.sum(SalesData.total_amount))
        ):
            product_type_analysis[product_type] = {
                "count": count,
                "total_amount": float(total_amount) if total_amount else 0,
            }

        # Análisis por categoría de prescripción
        prescription_analysis = {}
        for prescription_category, count, total_amount in (
            query.filter(cls.prescription_category.isnot(None))
            .join(SalesData)
            .group_by(cls.prescription_category)
            .values(
                cls.prescription_category,
                func.count(cls.id),
                func.sum(SalesData.total_amount),
            )
        ):
            prescription_analysis[prescription_category] = {
                "count": count,
                "total_amount": float(total_amount) if total_amount else 0,
            }

        return {
            "product_type_breakdown": product_type_analysis,
            "prescription_category_breakdown": prescription_analysis,
        }

    def update_generic_analysis(self, session):
        """
        Actualiza el análisis de genéricos para este registro

        Args:
            session: Sesión de SQLAlchemy
        """
        if not self.product_catalog:
            return

        # Calcular ahorro con genéricos
        savings_info = self.product_catalog.calculate_generic_savings(session)

        if savings_info:
            self.has_generic_alternative = True
            self.cheapest_generic_price = savings_info["cheapest_generic_price"]
            self.potential_generic_savings = savings_info["potential_savings"]
            self.generic_savings_percent = savings_info["savings_percent"]
        else:
            self.has_generic_alternative = False
            self.cheapest_generic_price = None
            self.potential_generic_savings = None
            self.generic_savings_percent = None

        session.commit()

    def mark_for_manual_review(self, reason: str = None):
        """
        Marca el registro para revisión manual

        Args:
            reason: Razón para la revisión manual
        """
        self.enrichment_status = "manual_review"
        self.manual_review_reason = reason
        self.last_updated = func.now()
