# backend/app/models/sales_data.py
import uuid

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    ForeignKey,
    Integer,
    Numeric,
    String,
    Text,
    Time,
)
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class SalesData(Base):
    """Modelo para datos de ventas normalizados"""

    __tablename__ = "sales_data"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Foreign keys
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        index=True,
        nullable=False,
    )
    upload_id = Column(
        UUID(as_uuid=True),
        ForeignKey("file_uploads.id", ondelete="CASCADE"),
        nullable=False,
        index=True,  # Crítico para /upload/history performance
    )

    # Información temporal
    # sale_date: DateTime - Mantener por compatibilidad con queries existentes que filtran
    #            por rango de fechas (start_date <= sale_date <= end_date).
    #            Incluye timestamp completo del ERP.
    # sale_time: Time - Hora específica extraída del timestamp para granularidad en constraint UNIQUE.
    #            Permite múltiples ventas del mismo producto en el mismo día a horas diferentes.
    #            Nullable para backward compatibility con datos históricos (Issue #273).
    sale_date = Column(DateTime, nullable=False, index=True)
    sale_time = Column(Time, nullable=True, index=True)
    year = Column(Integer, index=True)
    month = Column(Integer, index=True)
    day = Column(Integer)
    week = Column(Integer)  # Semana del año
    weekday = Column(Integer)  # Día de la semana (1=Lunes, 7=Domingo)

    # Información del producto
    product_name = Column(String(500))
    ean13 = Column(String(20), index=True)  # Código de barras EAN13 completo
    codigo_nacional = Column(String(20), index=True)  # Código Nacional (CN) - 6 dígitos
    # NOTA: laboratory, active_ingredient, therapeutic_group, category, product_type, is_generic
    # Ya están disponibles a través de las relaciones SalesEnrichment -> ProductCatalog
    # Eliminados para cumplir con normalización 3NF

    # Clasificación básica del producto (solo datos que no están en ProductCatalog)
    subcategory = Column(String(100))  # Subcategoría específica de la farmacia
    requires_prescription = Column(Boolean, default=False)
    is_narcotic = Column(Boolean, default=False)  # Estupefaciente
    is_refrigerated = Column(Boolean, default=False)  # Requiere frío

    # Cantidades
    quantity = Column(Integer, default=0)
    unit_of_measure = Column(String(20), default="units")  # 'units', 'boxes', 'ml', etc.

    # Precios (todos en la moneda de la farmacia)
    unit_price = Column(Numeric(10, 2))  # Precio unitario
    purchase_price = Column(Numeric(10, 2))  # Precio de coste/compra (PUC)
    sale_price = Column(Numeric(10, 2))  # Precio de venta al público (PVP)
    reference_price = Column(Numeric(10, 2))  # Precio de referencia (R.P.)
    gross_amount = Column(Numeric(10, 2))  # Importe bruto antes de descuentos
    total_amount = Column(Numeric(10, 2))  # Importe total de la línea

    # Descuentos
    discount_percentage = Column(Numeric(5, 2))
    discount_amount = Column(Numeric(10, 2))

    # Márgenes
    margin_amount = Column(Numeric(10, 2))  # Margen en euros
    margin_percentage = Column(Numeric(5, 2))  # Margen en porcentaje

    # Información del proveedor
    supplier = Column(String(255))
    supplier_code = Column(String(50))

    # Información de la venta
    operation_id = Column(String(50))  # ID de operación del ERP (Farmanager)
    invoice_number = Column(String(100))  # Número de ticket o factura
    client_type = Column(String(50))  # 'regular', 'mutual', 'insurance', etc.
    client_code = Column(String(50))  # Código de cliente si existe
    client_name = Column(String(200))  # Nombre del cliente
    employee_code = Column(String(50))  # Código del empleado que realizó la venta
    employee_name = Column(String(100))  # Nombre del empleado

    # Información de seguro/mutua
    insurance_company = Column(String(100))  # Mutua o seguro
    insurance_contribution = Column(Numeric(10, 2))  # Aportación del seguro
    patient_contribution = Column(Numeric(10, 2))  # Aportación del paciente
    contribution_type = Column(String(20))  # Tipo de aportación (R1, R2, etc.)

    # Flags especiales
    is_promotion = Column(Boolean, default=False)  # Producto en promoción
    is_return = Column(Boolean, default=False)  # Es una devolución
    is_expired = Column(Boolean, default=False)  # Producto caducado
    is_sample = Column(Boolean, default=False)  # Muestra gratuita
    is_financed = Column(Boolean, default=False)  # Tiene financiación/aportación
    is_cancelled = Column(Boolean, default=False)  # Operación cancelada

    # Metadatos
    raw_data = Column(Text)  # JSON con datos originales si necesario
    notes = Column(Text)  # Notas adicionales
    created_at = Column(DateTime, default=utc_now)

    # Relaciones con otras tablas
    pharmacy = relationship("Pharmacy", back_populates="sales_data")
    upload = relationship("FileUpload", back_populates="sales_data")
    enrichment = relationship(
        "SalesEnrichment",
        back_populates="sales_data",
        uselist=False,
        cascade="all, delete-orphan",
    )

    def __repr__(self):
        return f"<SalesData {self.product_name} - {self.sale_date}>"

    def to_dict(self, include_enrichment=False):
        """
        Convierte el modelo a diccionario para APIs

        Args:
            include_enrichment: Si incluir datos enriquecidos (via ProductCatalog)
        """
        result = {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "sale_date": self.sale_date.isoformat() if self.sale_date else None,
            "ean13": self.ean13,
            "codigo_nacional": self.codigo_nacional,
            "product_name": self.product_name,
            "quantity": self.quantity,
            "unit_price": float(self.unit_price) if self.unit_price else 0,
            "total_amount": float(self.total_amount) if self.total_amount else 0,
            "margin_percentage": (float(self.margin_percentage) if self.margin_percentage else 0),
            "is_return": self.is_return,
        }

        # Incluir datos enriquecidos si se solicita y están disponibles
        if include_enrichment and self.enrichment:
            enrichment_data = {}

            # Datos del catálogo de productos (via SalesEnrichment -> ProductCatalog)
            if self.enrichment.product_catalog:
                pc = self.enrichment.product_catalog
                enrichment_data.update(
                    {
                        "laboratory": pc.laboratory,
                        "active_ingredient": pc.active_ingredient,
                        "therapeutic_group": pc.therapeutic_group,
                        "is_generic": "genérico" in (pc.nomen_tipo_farmaco or "").lower(),
                        "requires_prescription": pc.requires_prescription,
                        "atc_code": pc.atc_code,
                    }
                )

            # Datos del enriquecimiento (categorización, análisis)
            enrichment_data.update(
                {
                    "product_type": self.enrichment.product_type,
                    "therapeutic_category": self.enrichment.therapeutic_category,
                    "prescription_category": self.enrichment.prescription_category,
                    "has_generic_alternative": self.enrichment.has_generic_alternative,
                    "potential_generic_savings": (
                        float(self.enrichment.potential_generic_savings)
                        if self.enrichment.potential_generic_savings
                        else None
                    ),
                    "enrichment_status": self.enrichment.enrichment_status,
                    "match_confidence": self.enrichment.match_confidence,
                }
            )

            result["enrichment"] = enrichment_data

        return result

    @property
    def profit(self):
        """Calcula el beneficio de la venta"""
        if self.margin_amount:
            return float(self.margin_amount)
        elif self.sale_price and self.purchase_price:
            return float((self.sale_price - self.purchase_price) * self.quantity)
        return 0
