# backend/app/models/role.py
import uuid

from sqlalchemy import Column, DateTime, String
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class Role(Base):
    """
    Modelo para roles del sistema con RBAC (Role-Based Access Control).

    Issue #348: Admin Panel Unificado con gestión de usuarios.

    Roles disponibles:
    - admin: Todos los permisos administrativos (6/6)
    - user: Sin permisos administrativos (0/6)

    Permisos disponibles:
    - manage_users: Crear, editar, eliminar usuarios y farmacias
    - delete_sales_data: Eliminar datos de ventas (GDPR Article 17)
    - delete_accounts: Eliminar cuentas completas (GDPR Article 17)
    - manage_backups: Crear y restaurar backups
    - view_audit_logs: Ver registros de auditoría
    - clear_cache: Limpiar cachés del sistema
    """

    __tablename__ = "roles"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Role information
    name = Column(String(50), unique=True, nullable=False, index=True)
    permissions = Column(JSONB, nullable=False)  # Array de permissions strings

    # Timestamps
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now)

    # Relationships
    users = relationship("User", back_populates="role_obj")

    def __repr__(self):
        return f"<Role {self.name} ({len(self.permissions)} permissions)>"

    def to_dict(self):
        """Convierte el modelo a diccionario para APIs"""
        return {
            "id": str(self.id),
            "name": self.name,
            "permissions": self.permissions,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None
        }

    def has_permission(self, permission: str) -> bool:
        """
        Verifica si el rol tiene un permiso específico.

        Args:
            permission: Permission constant from app.core.subscription_limits

        Returns:
            True if role has the permission, False otherwise
        """
        return permission in self.permissions

    @classmethod
    def get_permissions_count(cls, role_name: str) -> int:
        """
        Retorna la cantidad de permisos para un rol específico.

        Args:
            role_name: 'admin', 'user'

        Returns:
            Cantidad de permisos (0-6)
        """
        default_permissions = {
            'admin': 6,
            'user': 0
        }
        return default_permissions.get(role_name, 0)
