# backend/app/models/product_correction.py
"""
Modelo para almacenar correcciones humanas del clasificador.

Filosofía: Cada corrección no solo arregla un producto,
sino que alimenta el sistema de sugerencia de reglas automáticas.
"""

from datetime import datetime, timezone
from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    Float,
    ForeignKey,
    Integer,
    String,
    Text,
    Enum as SQLEnum,
)
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
import enum

from app.database import Base


class CorrectionType(str, enum.Enum):
    """Tipo de corrección realizada."""
    APPROVE = "approve"      # Sistema acertó, farmacéutico confirma
    CORRECT = "correct"      # Sistema falló, farmacéutico corrige
    OUTLIER = "outlier"      # Producto no encaja en ninguna categoría
    SKIP = "skip"            # Saltado temporalmente, va al final de la cola (Issue #457)


class PredictionSource(str, enum.Enum):
    """Fuente de la predicción original del sistema."""
    TIER1_SPECIFIC = "tier1_specific"
    TIER1_GENERIC = "tier1_generic"
    TIER2_RULES = "tier2_rules"
    BRAND = "brand"
    LLM = "llm"
    FALLBACK = "fallback"  # Cayó a "otros"


# Issue #457: Outlier investigation system
class OutlierReason(str, enum.Enum):
    """Razón por la que un producto se marcó como outlier."""
    PACK_PROMOCIONAL = "pack_promocional"         # "Pack 3x2", "Neceser regalo"
    PRODUCTO_DESCATALOGADO = "descatalogado"      # Producto ya no existe
    AMBIGUO = "ambiguo"                           # "Isdin Fusion" (¿Solar o Maquillaje?)
    REQUIERE_NUEVA_CATEGORIA = "nueva_categoria"  # Necesita categoría nueva
    OTRO = "otro"                                 # Casos no cubiertos


class OutlierStatus(str, enum.Enum):
    """Estado de investigación del outlier."""
    PENDING_REVIEW = "pending_review"             # Recién marcado, sin revisar
    UNDER_INVESTIGATION = "investigating"         # Dev lo está investigando
    RESOLVED_NEW_CATEGORY = "resolved_category"   # Se creó nueva categoría
    RESOLVED_CORRECTED = "resolved_corrected"     # Era clasificable, se corrigió
    TRUE_OUTLIER = "true_outlier"                 # Confirmado como no clasificable


class ProductCorrection(Base):
    """
    Registro de correcciones humanas para aprendizaje continuo.

    Cada registro contiene:
    - Qué predijo el sistema (y por qué)
    - Qué dijo el humano (ground truth)
    - Contexto para generar nuevas reglas
    """
    __tablename__ = "product_corrections"

    id = Column(Integer, primary_key=True, index=True)

    # === PRODUCTO ===
    sales_enrichment_id = Column(
        UUID(as_uuid=True),
        ForeignKey("sales_enrichment.id", ondelete="CASCADE"),
        nullable=False,
        index=True
    )
    product_name = Column(String(500), nullable=False)  # Snapshot para análisis

    # === PREDICCIÓN DEL SISTEMA ===
    predicted_category = Column(String(100), nullable=False)
    # Use String column to avoid SQLAlchemy enum name vs value mismatch
    # PostgreSQL has lowercase values but SQLAlchemy sends uppercase names
    prediction_source = Column(String(50), nullable=False)
    confidence_score = Column(Float, nullable=False)

    # === CORRECCIÓN HUMANA ===
    # Use String column to avoid SQLAlchemy enum name vs value mismatch
    correction_type = Column(String(20), nullable=False)
    corrected_category = Column(String(100), nullable=True)  # NULL si APPROVE

    # === CONTEXTO ===
    reviewer_notes = Column(Text, nullable=True)
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="SET NULL"),
        nullable=True
    )

    # === OUTLIER INVESTIGATION (Issue #457) ===
    # Solo se usan cuando correction_type == OUTLIER
    # Use String columns to avoid SQLAlchemy enum name vs value mismatch
    outlier_reason = Column(String(30), nullable=True)  # NULL si no es outlier
    outlier_status = Column(String(30), nullable=True, default=None)  # NULL si no es outlier
    outlier_resolved_at = Column(DateTime(timezone=True), nullable=True)
    outlier_resolution_notes = Column(Text, nullable=True)

    # === TIMESTAMPS ===
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False
    )

    # === ESTADO DE PROCESAMIENTO ===
    # ¿Ya se usó esta corrección para sugerir/crear reglas?
    processed_for_rules = Column(Boolean, default=False, nullable=False)
    processed_at = Column(DateTime(timezone=True), nullable=True)
    suggested_rule = Column(Text, nullable=True)  # Regla sugerida (si aplica)

    # === RELACIONES ===
    sales_enrichment = relationship(
        "SalesEnrichment",
        back_populates="corrections",
        lazy="selectin"
    )
    pharmacy = relationship("Pharmacy", lazy="selectin")

    def __repr__(self):
        return (
            f"<ProductCorrection(id={self.id}, "
            f"predicted={self.predicted_category}, "
            f"corrected={self.corrected_category}, "
            f"type={self.correction_type})>"
        )

    @property
    def was_correct(self) -> bool:
        """True si el sistema acertó (APPROVE)."""
        return self.correction_type == CorrectionType.APPROVE.value

    @property
    def is_actionable(self) -> bool:
        """True si esta corrección puede generar una regla."""
        return (
            self.correction_type == CorrectionType.CORRECT.value
            and not self.processed_for_rules
            and self.corrected_category is not None
        )

    @property
    def is_outlier(self) -> bool:
        """True si este registro es un outlier."""
        return self.correction_type == CorrectionType.OUTLIER.value

    @property
    def is_outlier_pending(self) -> bool:
        """True si es un outlier pendiente de investigación."""
        return (
            self.is_outlier
            and self.outlier_status in (
                OutlierStatus.PENDING_REVIEW.value,
                OutlierStatus.UNDER_INVESTIGATION.value,
                None  # Legacy: outliers sin status
            )
        )

    @property
    def is_outlier_resolved(self) -> bool:
        """True si es un outlier ya resuelto."""
        return (
            self.is_outlier
            and self.outlier_status in (
                OutlierStatus.RESOLVED_NEW_CATEGORY.value,
                OutlierStatus.RESOLVED_CORRECTED.value,
                OutlierStatus.TRUE_OUTLIER.value,
            )
        )
