# backend/app/models/product_cluster.py
"""
Modelo ProductCluster - Clusters jerárquicos con sistema de anclas

Issue #456: Pipeline de Clasificación Venta Libre (M2)

Este modelo representa clusters de productos para clasificación automática:
- Jerarquía de 4 niveles (10, 30, 100, 300 clusters)
- Sistema de anclas (centroides) para asignación de nuevos productos
- State machine: PROVISIONAL → LOCKED → ARCHIVED
- Validación por farmacéutico antes de usar en producción

Integración con ClassifierService:
- LOCKED clusters: Usados para clasificación automática (predict_fast < 500ms)
- PROVISIONAL clusters: Beta mode con confidence penalty 20%
- ARCHIVED clusters: Histórico, no usados para clasificación
"""

import uuid
from datetime import datetime, timezone
from enum import Enum
from typing import TYPE_CHECKING, List, Optional

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    Float,
    ForeignKey,
    Index,
    Integer,
    Numeric,
    String,
    Text,
)
from sqlalchemy import Enum as SQLEnum
from sqlalchemy.dialects.postgresql import ARRAY, UUID
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from .base import Base

if TYPE_CHECKING:
    from .sales_enrichment import SalesEnrichment


class ClusterState(str, Enum):
    """
    Estado del cluster en el ciclo de vida.

    PROVISIONAL: Cluster generado automáticamente, pendiente de validación.
                 Puede usarse en beta mode con confidence penalty.
    LOCKED: Validado por farmacéutico, usado para clasificación en producción.
    ARCHIVED: Deprecado o reemplazado. Solo para histórico.
    """
    PROVISIONAL = "provisional"
    LOCKED = "locked"
    ARCHIVED = "archived"


class ProductCluster(Base):
    """
    Cluster de productos para clasificación jerárquica.

    Cada cluster tiene:
    - Un centroide (embedding promedio de sus productos)
    - Un nivel en la jerarquía (0-3)
    - Un estado que determina si puede usarse para clasificación
    - Métricas de calidad (silhouette score, purity)

    Jerarquía de niveles:
    - Nivel 0: ~10 clusters (macro categorías: proteccion_solar, nutricion, etc.)
    - Nivel 1: ~30 clusters (sub-categorías: facial, corporal, infantil)
    - Nivel 2: ~100 clusters (familias de producto)
    - Nivel 3: ~300 clusters (grupos específicos)

    Ejemplo de uso:
    >>> cluster = session.query(ProductCluster).filter_by(
    ...     state=ClusterState.LOCKED,
    ...     hierarchy_level=2
    ... ).all()
    >>> nearest = classifier.find_nearest(embedding, cluster)
    """

    __tablename__ = "product_cluster"

    # === CLAVE PRIMARIA ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # === IDENTIFICACIÓN ===
    name = Column(
        String(200),
        nullable=False,
        comment="Nombre descriptivo del cluster (generado por LLM o manual)"
    )
    slug = Column(
        String(100),
        unique=True,
        index=True,
        nullable=False,
        comment="Identificador URL-friendly único (ej: 'fotoprotectores-faciales-spf50')"
    )
    description = Column(
        Text,
        nullable=True,
        comment="Descripción del cluster y productos típicos"
    )

    # === JERARQUÍA ===
    hierarchy_level = Column(
        Integer,
        nullable=False,
        index=True,
        comment="Nivel en la jerarquía: 0 (10 clusters), 1 (30), 2 (100), 3 (300)"
    )
    parent_cluster_id = Column(
        UUID(as_uuid=True),
        ForeignKey("product_cluster.id", ondelete="SET NULL"),
        nullable=True,
        index=True,
        comment="Cluster padre en nivel superior (None para nivel 0)"
    )

    # === CENTROIDE (ANCHOR EMBEDDING) ===
    centroid_embedding = Column(
        ARRAY(Float),
        nullable=True,
        comment="Embedding 384-dim del centroide (sentence-transformers)"
    )
    centroid_product_id = Column(
        UUID(as_uuid=True),
        nullable=True,
        comment="ID del producto más cercano al centroide (representante)"
    )
    centroid_product_name = Column(
        String(300),
        nullable=True,
        comment="Nombre del producto representante para visualización"
    )

    # === CLASIFICACIÓN ===
    primary_necesidad = Column(
        String(100),
        nullable=True,
        index=True,
        comment="NECESIDAD principal del cluster (ej: 'proteccion_solar')"
    )
    primary_subcategory = Column(
        String(100),
        nullable=True,
        index=True,
        comment="Subcategoría principal (ej: 'facial', 'corporal')"
    )
    top_brands = Column(
        ARRAY(String(100)),
        nullable=True,
        comment="Top 5 marcas más frecuentes en el cluster"
    )

    # === NOMBRE GENERADO POR LLM ===
    llm_generated_name = Column(
        String(200),
        nullable=True,
        comment="Nombre generado por LLM basado en productos del cluster"
    )
    llm_name_confidence = Column(
        Numeric(3, 2),
        nullable=True,
        comment="Confianza del LLM en el nombre generado (0.00-1.00)"
    )

    # === STATE MACHINE ===
    state = Column(
        SQLEnum(ClusterState, name='cluster_state_enum', create_type=False),
        nullable=False,
        default=ClusterState.PROVISIONAL,
        index=True,
        comment="Estado: provisional (auto), locked (validado), archived (deprecado)"
    )

    # === UMBRAL DE ASIGNACIÓN ===
    assignment_threshold = Column(
        Numeric(3, 2),
        default=0.85,
        nullable=False,
        comment="Umbral de similitud mínimo para asignar productos (0.00-1.00)"
    )

    # === VALIDACIÓN ===
    validated_by = Column(
        String(100),
        nullable=True,
        comment="Email o nombre del farmacéutico que validó"
    )
    validated_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha/hora de validación (cuando pasó a LOCKED)"
    )
    validation_notes = Column(
        Text,
        nullable=True,
        comment="Notas del farmacéutico sobre la validación"
    )

    # === MÉTRICAS DE CALIDAD ===
    silhouette_score = Column(
        Numeric(5, 4),
        nullable=True,
        comment="Silhouette score del cluster (-1 a 1, mayor = mejor separación)"
    )
    purity_score = Column(
        Numeric(5, 4),
        nullable=True,
        comment="Purity score: % productos con misma NECESIDAD (0-1)"
    )
    cohesion_score = Column(
        Numeric(5, 4),
        nullable=True,
        comment="Cohesión interna: distancia promedio al centroide (menor = mejor)"
    )

    # === ESTADÍSTICAS (DENORMALIZADAS) ===
    product_count = Column(
        Integer,
        default=0,
        comment="Número de productos asignados al cluster"
    )
    brand_count = Column(
        Integer,
        default=0,
        comment="Número de marcas diferentes en el cluster"
    )
    total_sales_amount = Column(
        Numeric(12, 2),
        default=0,
        comment="Suma total de ventas de productos del cluster"
    )

    # === TIMESTAMPS ===
    created_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        nullable=False,
        comment="Fecha de creación del cluster"
    )
    updated_at = Column(
        DateTime(timezone=True),
        onupdate=func.now(),
        nullable=True,
        comment="Última actualización"
    )

    # === TAXONOMY LABELING (Issue #462) ===
    composition_hash = Column(
        String(64),
        nullable=True,
        comment="SHA256 hash of sorted product_ids for change detection"
    )
    labeled_at = Column(
        DateTime(timezone=True),
        nullable=True,
        index=True,
        comment="Timestamp of last taxonomy labeling"
    )
    llm_version = Column(
        String(50),
        nullable=True,
        comment="LLM model version used for naming (e.g., gpt-4o-mini)"
    )

    # === RELACIONES ===
    parent = relationship(
        "ProductCluster",
        remote_side=[id],
        backref="children",
        foreign_keys=[parent_cluster_id]
    )

    # Relación con SalesEnrichment (productos asignados)
    # Se añadirá FK en SalesEnrichment: product_cluster_id
    sales_enrichments = relationship(
        "SalesEnrichment",
        back_populates="product_cluster",
        lazy="dynamic",
        foreign_keys="SalesEnrichment.product_cluster_id"
    )

    # === ÍNDICES COMPUESTOS ===
    __table_args__ = (
        Index('ix_product_cluster_state_level', 'state', 'hierarchy_level'),
        Index('ix_product_cluster_necesidad_state', 'primary_necesidad', 'state'),
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        return (
            f"<ProductCluster("
            f"id={self.id}, "
            f"name='{self.name}', "
            f"level={self.hierarchy_level}, "
            f"state={self.state.value}, "
            f"products={self.product_count}"
            f")>"
        )

    def to_dict(self) -> dict:
        """Convierte el cluster a diccionario para serialización."""
        return {
            "id": str(self.id),
            "name": self.name,
            "slug": self.slug,
            "description": self.description,
            "hierarchy_level": self.hierarchy_level,
            "parent_cluster_id": str(self.parent_cluster_id) if self.parent_cluster_id else None,
            "centroid_product_name": self.centroid_product_name,
            "primary_necesidad": self.primary_necesidad,
            "primary_subcategory": self.primary_subcategory,
            "top_brands": self.top_brands,
            "llm_generated_name": self.llm_generated_name,
            "llm_name_confidence": float(self.llm_name_confidence) if self.llm_name_confidence else None,
            "state": self.state.value,
            "assignment_threshold": float(self.assignment_threshold) if self.assignment_threshold else 0.85,
            "validated_by": self.validated_by,
            "validated_at": self.validated_at.isoformat() if self.validated_at else None,
            "validation_notes": self.validation_notes,
            "silhouette_score": float(self.silhouette_score) if self.silhouette_score else None,
            "purity_score": float(self.purity_score) if self.purity_score else None,
            "cohesion_score": float(self.cohesion_score) if self.cohesion_score else None,
            "product_count": self.product_count,
            "brand_count": self.brand_count,
            "total_sales_amount": float(self.total_sales_amount) if self.total_sales_amount else 0,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
            # Taxonomy Labeling (Issue #462)
            "composition_hash": self.composition_hash,
            "labeled_at": self.labeled_at.isoformat() if self.labeled_at else None,
            "llm_version": self.llm_version,
        }

    def lock(self, validated_by: str, notes: str = None) -> None:
        """
        Marca el cluster como LOCKED (validado) para uso en producción.

        Args:
            validated_by: Email o nombre del farmacéutico
            notes: Notas opcionales sobre la validación
        """
        if self.state == ClusterState.ARCHIVED:
            raise ValueError("Cannot lock an archived cluster")

        self.state = ClusterState.LOCKED
        self.validated_by = validated_by
        self.validated_at = datetime.now(timezone.utc)
        self.validation_notes = notes

    def archive(self, reason: str = None) -> None:
        """
        Archiva el cluster (deprecado).

        Args:
            reason: Razón del archivo
        """
        self.state = ClusterState.ARCHIVED
        self.validation_notes = f"Archived: {reason}" if reason else self.validation_notes

    def unlock(self) -> None:
        """
        Devuelve el cluster a estado PROVISIONAL para re-validación.

        Solo permitido desde LOCKED.
        """
        if self.state != ClusterState.LOCKED:
            raise ValueError("Can only unlock a locked cluster")

        self.state = ClusterState.PROVISIONAL
        # Mantener historial de quién validó previamente

    def is_usable_for_classification(self) -> bool:
        """
        Verifica si el cluster puede usarse para clasificación.

        Returns:
            True si LOCKED o si es el único disponible (beta mode)
        """
        return self.state in (ClusterState.LOCKED, ClusterState.PROVISIONAL)

    def get_confidence_penalty(self) -> float:
        """
        Retorna el penalty de confianza según el estado.

        Returns:
            1.0 para LOCKED, 0.8 para PROVISIONAL (20% penalty)
        """
        if self.state == ClusterState.LOCKED:
            return 1.0
        elif self.state == ClusterState.PROVISIONAL:
            return 0.8  # 20% penalty en beta mode
        else:
            return 0.0  # ARCHIVED no usable

    def update_stats(self, session) -> None:
        """
        Actualiza estadísticas denormalizadas del cluster.

        Args:
            session: Sesión de SQLAlchemy
        """
        from sqlalchemy import func as sqla_func
        from .sales_data import SalesData
        from .sales_enrichment import SalesEnrichment

        stats = (
            session.query(
                sqla_func.count(sqla_func.distinct(SalesData.product_name)).label("products"),
                sqla_func.count(sqla_func.distinct(SalesEnrichment.detected_brand)).label("brands"),
                sqla_func.sum(SalesData.total_amount).label("amount"),
            )
            .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
            .filter(SalesEnrichment.product_cluster_id == self.id)
            .first()
        )

        if stats:
            self.product_count = stats.products or 0
            self.brand_count = stats.brands or 0
            self.total_sales_amount = stats.amount or 0


# Nota: La FK product_cluster_id en SalesEnrichment se añade en la migración
