# backend/app/models/product_catalog_vet.py
"""
Modelo ProductCatalogVet - Catálogo de productos veterinarios desde CIMAVet
Issue #354: Integración CIMAVet para completar cobertura farmacéutica
"""

import uuid
from datetime import datetime, timezone

from sqlalchemy import Boolean, Column, DateTime, Index, Integer, Numeric, String
from sqlalchemy.dialects.postgresql import ARRAY

from .base import Base


def utc_now():
    """Helper para timestamps UTC consistentes"""
    return datetime.now(timezone.utc)


class ProductCatalogVet(Base):
    """
    Catálogo de productos veterinarios desde CIMAVet (AEMPS).

    Tabla separada para productos de uso exclusivo veterinario.
    Consultada SOLO cuando producto NO existe en CIMA/Nomenclator.

    Issue #354: Integración CIMAVet
    - ~3,500 productos veterinarios oficiales
    - Sincronización cada 10 días (productos vet cambian menos frecuentemente)
    - Resiliencia con circuit breaker (lecciones Issue #114)

    Relación con product_catalog:
    - Tabla SEPARADA (no integrada) para minimizar riesgo breaking changes
    - Al encontrar producto aquí, se crea en product_catalog con:
      * xfarma_prescription_category = "VETERINARIA"
      * cima_uso_veterinario = True
      * data_sources = "CIMAVet"
      * cima_requiere_receta = vet_requiere_receta (para clasificación venta)

    Cascada de enriquecimiento (actualizada Issue #354):
    1. CIMA (medicamentos uso humano)
    2. Nomenclator (catálogo oficial Ministerio)
    3. CIMAVet (medicamentos uso veterinario) ← NUEVO
    4. manual_review
    """

    __tablename__ = "product_catalog_vet"
    __table_args__ = {"extend_existing": True}  # Thread-safe para Render multi-worker

    # === CLAVE PRIMARIA ===
    national_code = Column(String(20), primary_key=True, index=True)
    # Prefijo "VET-" obligatorio para evitar colisión con códigos humanos
    # Ejemplo: "VET-123456" para código nacional 123456 de CIMAVet

    # === DATOS BÁSICOS CIMAVet ===
    vet_nombre_comercial = Column(String(500), nullable=False, index=True)
    # Nombre comercial del producto veterinario

    vet_principios_activos = Column(ARRAY(String), nullable=True)
    # Lista de principios activos
    # Ejemplo: ["Amoxicilina", "Ácido Clavulánico"]

    vet_laboratorio_titular = Column(String(200), nullable=True, index=True)
    # Laboratorio titular del registro
    # Índice para análisis de laboratorios veterinarios

    vet_forma_farmaceutica = Column(String(200), nullable=True)
    # Forma farmacéutica: "Comprimidos", "Suspensión oral", "Inyectable", etc.

    vet_via_administracion = Column(String(100), nullable=True)
    # Vía de administración: "Oral", "Intramuscular", "Tópica", etc.

    # === ESPECÍFICO VETERINARIA ===
    vet_especies_destino = Column(ARRAY(String), nullable=True)
    # Especies animales destino
    # Ejemplo: ["Perros", "Gatos"] o ["Bovino", "Ovino"]
    # Índice GIN para búsquedas eficientes en arrays

    vet_condiciones_prescripcion = Column(String(50), nullable=True)
    # Condiciones de prescripción según CIMAVet
    # Ejemplo: "Receta veterinaria", "Uso bajo prescripción", etc.

    # 🆕 CRÍTICO: Campo para determinar tipo de venta
    vet_requiere_receta = Column(Boolean, nullable=True, index=True)
    # True: Requiere receta veterinaria → product_type = "prescription"
    # False: Venta libre → product_type = "venta_libre"
    # NULL: No determinado → asumir True por seguridad
    #
    # IMPORTANTE: Este campo se mapea a cima_requiere_receta al crear
    # el producto en product_catalog, ya que _derive_product_type()
    # usa ese campo para clasificar productos VETERINARIOS.
    #
    # Excepción: VETERINARIA es la ÚNICA categoría donde tener la categoría
    # NO determina automáticamente el tipo de venta (prescription vs venta_libre).

    # === PRECIOS VETERINARIA ===
    vet_pvp = Column(Numeric(10, 4), nullable=True)
    # Precio venta público con IVA (si disponible en CIMAVet)
    # Nota: No todos los productos veterinarios tienen PVP regulado

    # === ESTADO Y REGISTRO ===
    vet_estado_registro = Column(String(50), nullable=True)
    # Estado del registro: "AUTORIZADO", "SUSPENDIDO", "REVOCADO", etc.

    vet_numero_registro = Column(String(100), nullable=True)
    # Número de registro oficial en AEMPS
    # Ejemplo: "EU/2/15/123/001"

    # === METADATOS ===
    data_source = Column(String(50), default="CIMAVet", nullable=False)
    # SIEMPRE "CIMAVet" para esta tabla
    # Permite distinguir origen al transferir a product_catalog

    enrichment_confidence = Column(Integer, default=90)
    # Alta confianza (90) - datos oficiales AEMPS
    # Similar a nomenclator_local (confianza 95)

    # === TIMESTAMPS ===
    created_at = Column(DateTime(timezone=True), default=utc_now, nullable=False)
    # Fecha de creación en nuestra BD

    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now, nullable=False)
    # Fecha de última actualización

    # === ÍNDICES COMPUESTOS ===
    __table_args__ = (
        # Índice para búsquedas por laboratorio
        Index("idx_vet_laboratorio", "vet_laboratorio_titular"),
        # Índice para filtrar por requiere receta
        Index("idx_vet_requiere_receta", "vet_requiere_receta"),
        # Índice GIN para búsquedas en array de especies
        Index("idx_vet_especies", "vet_especies_destino", postgresql_using="gin"),
        # Índice GIN para búsquedas en array de principios activos
        Index("idx_vet_principios_activos", "vet_principios_activos", postgresql_using="gin"),
        {"extend_existing": True},  # Thread-safe para Render multi-worker
    )

    def __repr__(self):
        return (
            f"<ProductCatalogVet(national_code='{self.national_code}', "
            f"nombre='{self.vet_nombre_comercial[:30]}...', "
            f"laboratorio='{self.vet_laboratorio_titular}', "
            f"requiere_receta={self.vet_requiere_receta})>"
        )

    def to_dict(self):
        """
        Convierte el modelo a diccionario para serialización.
        Útil para cache y APIs.
        """
        return {
            "national_code": self.national_code,
            "vet_nombre_comercial": self.vet_nombre_comercial,
            "vet_principios_activos": self.vet_principios_activos,
            "vet_laboratorio_titular": self.vet_laboratorio_titular,
            "vet_forma_farmaceutica": self.vet_forma_farmaceutica,
            "vet_via_administracion": self.vet_via_administracion,
            "vet_especies_destino": self.vet_especies_destino,
            "vet_condiciones_prescripcion": self.vet_condiciones_prescripcion,
            "vet_requiere_receta": self.vet_requiere_receta,
            "vet_pvp": float(self.vet_pvp) if self.vet_pvp else None,
            "vet_estado_registro": self.vet_estado_registro,
            "vet_numero_registro": self.vet_numero_registro,
            "data_source": self.data_source,
            "enrichment_confidence": self.enrichment_confidence,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }
