# backend/app/models/product_catalog_venta_libre.py
"""
Catálogo interno de productos Venta Libre / Parafarmacia.

Issue #457: Este catálogo es el equivalente a ProductCatalog (CIMA) pero para
productos que NO tienen código nacional. Se puebla automáticamente desde las
ventas de todas las farmacias y se enriquece con clasificación ML + validación humana.

Issue #473: Fuzzy matching para detectar duplicados semánticos usando tokens
ordenados alfabéticamente y rapidfuzz.

Arquitectura:
    ProductCatalog (CIMA)           → Medicamentos con CN (fuente externa)
    ProductCatalogVet (CIMAVET)     → Veterinaria con CN (fuente externa)
    ProductCatalogVentaLibre        → Parafarmacia sin CN (fuente INTERNA)

El flujo de enriquecimiento:
    1. Venta llega (product_name del ERP)
    2. Si tiene CN → buscar en ProductCatalog/ProductCatalogVet
    3. Si NO tiene CN → buscar en ProductCatalogVentaLibre
       - Si existe → usar clasificación cacheada
       - Si no existe → clasificar + crear entrada
    4. SalesEnrichment referencia el catálogo correspondiente
"""

import re
import uuid
from datetime import datetime, timezone

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    Float,
    Index,
    Integer,
    String,
    Text,
)
from sqlalchemy.dialects.postgresql import ARRAY, UUID
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from .base import Base


class ProductCatalogVentaLibre(Base):
    """
    Catálogo maestro de productos venta libre/parafarmacia.

    Cada producto único (por nombre normalizado) tiene UNA entrada aquí.
    Las ventas (SalesEnrichment) referencian este catálogo en lugar de
    almacenar la clasificación duplicada.
    """

    __tablename__ = "product_catalog_venta_libre"

    # Umbral de confianza para P2 (sincronizado con FeedbackServiceV2)
    # 2025-12-20: Bajado de 0.85 → 0.75 → 0.60
    LOW_CONFIDENCE_THRESHOLD = 0.60

    # === IDENTIFICACIÓN ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Nombre normalizado (lowercase, sin espacios extra) - CLAVE ÚNICA
    product_name_normalized = Column(
        String(500),
        unique=True,
        nullable=False,
        index=True,
        comment="Nombre normalizado para matching (lowercase, trim)"
    )

    # Nombre para mostrar (primer nombre visto, más legible)
    product_name_display = Column(
        String(500),
        nullable=False,
        comment="Nombre original para UI (primer nombre visto)"
    )

    # Variantes del nombre vistas (para fuzzy matching futuro)
    product_name_variants = Column(
        ARRAY(String(500)),
        default=[],
        comment="Lista de variantes del nombre vistas en distintas farmacias"
    )

    # Tokens ordenados para fuzzy matching (Issue #473)
    product_name_tokens = Column(
        String(500),
        nullable=True,
        index=True,
        comment="Tokens ordenados alfabéticamente para fuzzy matching"
    )

    # === CÓDIGOS DE IDENTIFICACIÓN (Issue #474) ===
    ean13 = Column(
        String(13),
        nullable=True,
        index=True,
        comment="EAN-13 principal del producto (validado con checksum)"
    )

    # Issue #481: Multi-EAN synonym support
    ean_codes = Column(
        ARRAY(String(20)),
        default=[],
        comment="Todos los EANs válidos del producto (multi-EAN synonym support)"
    )

    cn_codes = Column(
        ARRAY(String(20)),
        default=[],
        comment="Códigos nacionales vistos (6-7 dígitos, pueden variar por farmacia)"
    )

    # === TRACKING DE FUENTES (Issue #474) ===
    pharmacy_ids_seen = Column(
        ARRAY(UUID(as_uuid=True)),
        default=[],
        comment="UUIDs de farmacias que han vendido este producto"
    )

    first_pharmacy_id = Column(
        UUID(as_uuid=True),
        nullable=True,
        comment="UUID de la primera farmacia que reportó el producto"
    )

    # === CLASIFICACIÓN ML (equivalente a ATC en CIMA) ===
    ml_category = Column(
        String(100),
        index=True,
        comment="Categoría predicha (necesidad_especifica de symptom_taxonomy)"
    )

    ml_confidence = Column(
        Float,
        comment="Confianza de la predicción (0.0 - 1.0)"
    )

    prediction_source = Column(
        String(50),
        comment="Fuente: tier1_specific, tier1_generic, brand, llm, human"
    )

    detected_brand = Column(
        String(100),
        index=True,
        comment="Marca detectada (ISDIN, CERAVE, etc.)"
    )

    # === CLASIFICACIÓN ML LEVEL 2 (Issue #505) ===
    # Sub-categorías para análisis granular: DERMOCOSMÉTICA (7 L2), SUPLEMENTOS (6 L2), HIGIENE BUCAL (5 L2)
    ml_subcategory_l2 = Column(
        String(100),
        nullable=True,
        index=True,
        comment="Subcategoría L2 (higiene_limpieza, solar_facial, tratamiento_avanzado, etc.)"
    )

    ml_subcategory_l2_confidence = Column(
        Float,
        nullable=True,
        default=0.0,
        comment="Confianza de predicción L2 (0.0-1.0)"
    )

    subcategory_l2_source = Column(
        String(50),
        nullable=True,
        comment="Fuente L2: tier1_keywords, groq, human"
    )

    # === VALIDACIÓN HUMANA ===
    human_verified = Column(
        Boolean,
        default=False,
        index=True,
        comment="True si un humano ha validado/corregido la clasificación"
    )

    verified_category = Column(
        String(100),
        comment="Categoría final tras validación humana (puede diferir de ml_category)"
    )

    verified_at = Column(
        DateTime(timezone=True),
        comment="Fecha de verificación humana"
    )

    reviewer_notes = Column(
        Text,
        comment="Notas del revisor durante validación"
    )

    # === ESTADÍSTICAS DE USO ===
    first_seen_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        comment="Primera vez que se vio este producto"
    )

    last_seen_at = Column(
        DateTime(timezone=True),
        comment="Última vez que se vio este producto en una venta"
    )

    total_sales_count = Column(
        Integer,
        default=0,
        comment="Número total de ventas de este producto (todas las farmacias)"
    )

    pharmacies_count = Column(
        Integer,
        default=0,
        comment="En cuántas farmacias distintas se ha vendido"
    )

    # === UMAP VISUALIZATION (Issue #458) ===
    umap_x = Column(
        Float,
        nullable=True,
        comment="X coordinate in UMAP 2D space"
    )

    umap_y = Column(
        Float,
        nullable=True,
        comment="Y coordinate in UMAP 2D space"
    )

    umap_version = Column(
        String(20),
        nullable=True,
        comment="UMAP model version (e.g., v1.0)"
    )

    # === METADATA ===
    is_active = Column(
        Boolean,
        default=True,
        comment="False para productos descatalogados o erróneos"
    )

    is_outlier = Column(
        Boolean,
        default=False,
        comment="True si es pack/descatalogado/ambiguo (no clasificable)"
    )

    outlier_reason = Column(
        String(50),
        comment="Razón de outlier: pack_promocional, descatalogado, ambiguo, etc."
    )

    # === DUPLICATE MANAGEMENT (Issue #477) ===
    duplicate_review_status = Column(
        String(20),
        nullable=True,
        index=True,
        comment="Status: pending_review, confirmed_different, merged"
    )

    duplicate_group_id = Column(
        UUID(as_uuid=True),
        nullable=True,
        index=True,
        comment="ID del grupo de duplicados potenciales (mismo grupo = candidatos a merge)"
    )

    merged_into_id = Column(
        UUID(as_uuid=True),
        nullable=True,
        index=True,  # Issue #477: Optimizar queries de merge lookup
        comment="ID del producto primario si este fue mergeado"
    )

    merged_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha de merge"
    )

    merged_by = Column(
        UUID(as_uuid=True),
        nullable=True,
        comment="UUID del usuario que realizó el merge"
    )

    # === TIMESTAMPS ===
    created_at = Column(
        DateTime(timezone=True),
        server_default=func.now()
    )

    updated_at = Column(
        DateTime(timezone=True),
        onupdate=func.now()
    )

    # === RELATIONSHIPS ===
    # SalesEnrichment references this catalog
    sales_enrichments = relationship(
        "SalesEnrichment",
        back_populates="venta_libre_product",
        foreign_keys="SalesEnrichment.venta_libre_product_id"
    )

    # === INDEXES ===
    __table_args__ = (
        # Índice para búsqueda por categoría + no verificado (cola de feedback)
        Index(
            "ix_venta_libre_feedback_queue",
            "ml_category",
            "human_verified",
            "is_active",
        ),
        # Índice para búsqueda por marca
        Index(
            "ix_venta_libre_brand",
            "detected_brand",
        ),
        # Índice para productos más vendidos
        Index(
            "ix_venta_libre_sales_count",
            "total_sales_count",
        ),
        # Índice para UMAP visualization (Issue #458)
        Index(
            "ix_venta_libre_umap_coords",
            "umap_version",
            postgresql_where="umap_x IS NOT NULL",
        ),
        # Índice GIN para búsqueda por CN (Issue #474)
        Index(
            "ix_venta_libre_cn_codes_gin",
            "cn_codes",
            postgresql_using="gin",
        ),
        # Índice GIN para búsqueda por EAN (Issue #481: multi-EAN synonym support)
        Index(
            "ix_venta_libre_ean_codes_gin",
            "ean_codes",
            postgresql_using="gin",
        ),
    )

    def __repr__(self):
        return f"<ProductCatalogVentaLibre(id={self.id}, name='{self.product_name_display[:30]}...', category='{self.ml_category}')>"

    @property
    def effective_category(self) -> str:
        """Devuelve la categoría efectiva (verificada > predicha)."""
        if self.human_verified and self.verified_category:
            return self.verified_category
        return self.ml_category

    @property
    def needs_review(self) -> bool:
        """True si el producto necesita revisión humana."""
        if self.human_verified:
            return False
        if self.is_outlier:
            return False
        # P1: Categorías fallback
        fallback_categories = {"otros", "parafarmacia_otros", "unknown", "sin_clasificar"}
        if self.ml_category in fallback_categories:
            return True
        # P2: Baja confianza
        if self.ml_confidence and self.ml_confidence < self.LOW_CONFIDENCE_THRESHOLD:
            return True
        return False

    @classmethod
    def normalize_product_name(cls, name: str) -> str:
        """Normaliza el nombre del producto para matching exacto."""
        if not name:
            return ""
        # Lowercase, strip, collapse whitespace
        normalized = name.lower().strip()
        normalized = re.sub(r'\s+', ' ', normalized)
        return normalized

    # Diccionario de normalización de unidades farmacéuticas (Issue #473)
    # Mapea variantes a forma canónica
    UNIT_NORMALIZATION = {
        # Unidades
        'unidades': 'u', 'unidad': 'u', 'uds': 'u', 'ud': 'u',
        # Comprimidos/cápsulas
        'comprimidos': 'comp', 'comprimido': 'comp', 'caps': 'comp',
        'capsulas': 'comp', 'capsula': 'comp', 'tabletas': 'comp', 'tableta': 'comp',
        # Sobres
        'sobres': 'sobre', 'sachets': 'sobre', 'sachet': 'sobre',
        # Ampollas
        'ampollas': 'amp', 'ampolla': 'amp', 'viales': 'amp', 'vial': 'amp',
        # Mililitros (ya están normalizados)
        'mililitros': 'ml', 'mililitro': 'ml',
        # Gramos
        'gramos': 'g', 'gramo': 'g', 'gr': 'g',
        # Miligramos
        'miligramos': 'mg', 'miligramo': 'mg',
        # Litros
        'litros': 'l', 'litro': 'l',
        # Sticks
        'sticks': 'stick',
    }

    @classmethod
    def normalize_for_matching(cls, name: str) -> str:
        """
        Normalización agresiva para detectar duplicados semánticos (Issue #473).

        Proceso:
        1. Lowercase + strip
        2. Eliminar acentos (unidecode)
        3. Tokenizar (solo alfanuméricos)
        4. Normalizar unidades (usando UNIT_NORMALIZATION)
        5. Fusionar número + unidad (50 u → 50u, 100ml → 100ml)
        6. Ordenar tokens alfabéticamente

        IMPORTANTE: Los números se preservan exactamente para evitar falsos positivos.
        "180g" ≠ "120g" - esto se valida en ProductMatchingService.

        Ejemplo:
            "TIRAS CONTOUR NEXT 50U"   → "50u contour next tiras"
            "CONTOUR NEXT TIRAS 50 U"  → "50u contour next tiras"
            → ¡MATCH!

        Returns:
            String con tokens ordenados alfabéticamente
        """
        if not name:
            return ""

        try:
            from unidecode import unidecode
        except ImportError:
            # Fallback si unidecode no está instalado
            def unidecode(x):
                return x

        # 1. Lowercase + strip
        normalized = name.lower().strip()

        # 2. Eliminar acentos
        normalized = unidecode(normalized)

        # 3. Tokenizar (solo alfanuméricos)
        tokens = re.findall(r'\w+', normalized)

        # 4 & 5. Normalizar unidades y fusionar número + unidad
        normalized_tokens = []
        i = 0
        while i < len(tokens):
            token = tokens[i]

            # Si es un número y el siguiente es una unidad, fusionar
            if token.isdigit() and i + 1 < len(tokens):
                next_token = tokens[i + 1]
                # Normalizar unidad usando diccionario
                normalized_unit = cls.UNIT_NORMALIZATION.get(next_token, next_token)
                # Verificar si es una unidad conocida
                if next_token in cls.UNIT_NORMALIZATION or next_token in (
                    'u', 'ml', 'mg', 'g', 'l', 'comp', 'sobre', 'amp', 'stick'
                ):
                    normalized_tokens.append(f"{token}{normalized_unit}")
                    i += 2
                    continue

            # Normalizar token individual usando diccionario
            token = cls.UNIT_NORMALIZATION.get(token, token)
            # Eliminar puntos finales (50u. → 50u)
            token = re.sub(r'(\d+\w*)\.$', r'\1', token)
            normalized_tokens.append(token)
            i += 1

        # 6. Ordenar alfabéticamente
        sorted_tokens = sorted(normalized_tokens)

        return ' '.join(sorted_tokens)

    @staticmethod
    def is_valid_ean13(code: str) -> bool:
        """
        Valida EAN-13 con dígito de control (checksum).

        El algoritmo EAN-13:
        1. Suma dígitos en posiciones impares (1,3,5...) × 1
        2. Suma dígitos en posiciones pares (2,4,6...) × 3
        3. Dígito de control = (10 - (suma % 10)) % 10
        """
        if not code or len(code) != 13 or not code.isdigit():
            return False

        total = sum(
            int(digit) * (1 if i % 2 == 0 else 3)
            for i, digit in enumerate(code[:12])
        )
        check_digit = (10 - (total % 10)) % 10
        return int(code[12]) == check_digit

    @staticmethod
    def extract_cn_from_ean(ean: str) -> str | None:
        """
        Extrae código nacional de un EAN-13 español.

        Los EAN españoles de farmacia tienen formato: 8470006XXXXXX
        donde XXXXXX es el código nacional de 6 dígitos.
        """
        if not ean or len(ean) < 12:
            return None
        if ean.startswith("847000"):
            return ean[6:12]
        return None

    def add_pharmacy_source(
        self,
        pharmacy_id: "uuid.UUID",
        product_name: str | None = None,
        cn: str | None = None,
        ean: str | None = None
    ) -> bool:
        """
        Añade una farmacia como fuente de este producto.

        Actualiza:
        - pharmacy_ids_seen: añade el UUID si no existe
        - pharmacies_count: recalcula
        - product_name_variants: añade nombre si es diferente
        - cn_codes: añade CN si no existe
        - ean_codes: añade EAN si es válido y no existe (Issue #481)
        - ean13: establece como principal si es válido y no teníamos uno

        Returns:
            True si se modificó algo, False si ya estaba todo
        """
        modified = False

        # Actualizar pharmacy_ids_seen
        current_pharmacies = list(self.pharmacy_ids_seen or [])
        if pharmacy_id not in current_pharmacies:
            current_pharmacies.append(pharmacy_id)
            self.pharmacy_ids_seen = current_pharmacies
            self.pharmacies_count = len(current_pharmacies)

            # Primera farmacia
            if self.first_pharmacy_id is None:
                self.first_pharmacy_id = pharmacy_id

            modified = True

        # Añadir variante de nombre si es diferente
        if product_name:
            current_variants = list(self.product_name_variants or [])
            if product_name not in current_variants and len(current_variants) < 10:
                current_variants.append(product_name)
                self.product_name_variants = current_variants
                modified = True

        # Añadir CN si es nuevo y válido (6-7 dígitos)
        if cn and len(cn) >= 6 and len(cn) <= 7 and cn.isdigit():
            current_cns = list(self.cn_codes or [])
            if cn not in current_cns:
                current_cns.append(cn)
                self.cn_codes = current_cns
                modified = True

        # Issue #481: Añadir EAN al array ean_codes (multi-EAN synonym support)
        # Límite de 20 EANs por producto (productos raramente tienen más de 5)
        if ean and self.is_valid_ean13(ean):
            current_eans = list(self.ean_codes or [])
            if ean not in current_eans and len(current_eans) < 20:
                current_eans.append(ean)
                self.ean_codes = current_eans
                modified = True

            # Establecer ean13 principal si no teníamos uno
            if self.ean13 is None:
                self.ean13 = ean

            # También extraer CN del EAN si aplica
            extracted_cn = self.extract_cn_from_ean(ean)
            if extracted_cn:
                current_cns = list(self.cn_codes or [])
                if extracted_cn not in current_cns:
                    current_cns.append(extracted_cn)
                    self.cn_codes = current_cns

        return modified
