# backend/app/models/product_catalog.py
"""
Modelo ProductCatalog - Catálogo maestro con prefijos claros
Separación clara entre datos de CIMA (cima_) y Nomenclator (nomen_)
"""

import logging
import uuid

from sqlalchemy import Boolean, Column, DateTime, Enum as SQLEnum, Index, Integer, Numeric, String, event, inspect
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.sql import func

from .base import Base
from .enums import PrescriptionCategory

logger = logging.getLogger(__name__)


class ProductCatalog(Base):
    """
    Catálogo maestro de productos farmacéuticos con datos enriquecidos

    Estructura simplificada con prefijos claros:
    - cima_: Datos del Centro de Información de Medicamentos (AEMPS)
    - nomen_: Datos del Nomenclator de Facturación (Ministerio Sanidad)
    """

    __tablename__ = "product_catalog"

    # === CAMPOS BASE ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Identificadores principales
    national_code = Column(
        String(20), unique=True, nullable=True, index=True
    )  # Código Nacional (campo clave sin prefijo)

    # === CAMPOS NOMENCLATOR (nomen_) ===
    # Información básica del producto
    nomen_nombre = Column(String(500), nullable=True, index=True)  # Nombre del producto farmacéutico
    nomen_laboratorio = Column(String(200), nullable=True, index=True)  # Nombre del laboratorio ofertante
    nomen_codigo_laboratorio = Column(
        String(20), nullable=True, index=True
    )  # Código del laboratorio ofertante (nomenclator oficial)
    nomen_tipo_farmaco = Column(String(50), nullable=True)  # Tipo de fármaco
    nomen_principio_activo = Column(String(500), nullable=True, index=True)  # Principio activo o asociación

    # Información económica
    nomen_pvp = Column(Numeric(10, 4), nullable=True)  # Precio venta al público con IVA
    nomen_precio_referencia = Column(Numeric(10, 4), nullable=True)  # Precio de referencia
    nomen_aportacion_usuario = Column(Numeric(10, 4), nullable=True)  # Aportación del beneficiario
    nomen_pvl_calculado = Column(
        Numeric(10, 4), nullable=True
    )  # PVL calculado desde PVP (para productos ALTA con homogéneo)

    # Conjuntos homogéneos (genéricos)
    nomen_codigo_homogeneo = Column(String(50), nullable=True, index=True)  # Código agrupación homogénea
    nomen_nombre_homogeneo = Column(String(200), nullable=True)  # Nombre agrupación homogénea

    # Estado y fechas
    nomen_estado = Column(String(50), nullable=True, index=True)  # Estado comercialización
    nomen_fecha_alta = Column(DateTime, nullable=True)  # Fecha alta nomenclator
    nomen_fecha_baja = Column(DateTime, nullable=True)  # Fecha baja nomenclator

    # === CAMPOS CIMA (cima_) ===
    # Identificador CIMA
    cima_nregistro = Column(String(50), nullable=True, index=True)  # Número de registro del medicamento (para ATCs)

    # Descripción clínica
    cima_vmp = Column(String(500), nullable=True)  # VMP (Virtual Medicinal Product)
    cima_vmpp = Column(String(500), nullable=True)  # VMPP (Virtual Medicinal Product Pack)

    # Información regulatoria
    cima_nombre_comercial = Column(String(500), nullable=True, index=True)  # Nombre comercial
    cima_atc_codes = Column(JSONB, nullable=True)  # Lista de códigos ATC
    cima_atc_code = Column(String(20), nullable=True, index=True)  # Código ATC principal
    cima_grupo_terapeutico = Column(String(200), nullable=True)  # Grupo terapéutico
    cima_principios_activos = Column(JSONB, nullable=True)  # Lista de principios activos
    cima_forma_farmaceutica = Column(String(200), nullable=True)  # Forma farmacéutica
    cima_via_administracion = Column(String(100), nullable=True)  # Vía administración
    cima_dosis = Column(String(200), nullable=True)  # Dosis/concentración

    # Laboratorio y autorización
    cima_laboratorio_titular = Column(String(200), nullable=True, index=True)  # Laboratorio titular
    cima_condiciones_prescripcion = Column(String(200), nullable=True)  # Condiciones de prescripción
    cima_estado_registro = Column(String(50), nullable=True)  # Estado de registro/comercialización

    # Clasificación del producto
    cima_requiere_receta = Column(Boolean, nullable=True, default=None)  # Requiere receta
    cima_es_generico = Column(Boolean, nullable=True, default=False, index=True)  # Es genérico
    cima_uso_veterinario = Column(Boolean, nullable=True, default=False)  # Uso veterinario

    # === CAMPOS XFARMA (xfarma_) ===
    # Clasificación de prescripción (Issue #16 - Fase 1)
    xfarma_prescription_category = Column(
        SQLEnum(PrescriptionCategory, name='prescription_category_enum', values_callable=lambda x: [e.value for e in x]),
        nullable=True,
        index=True
    )  # Clasificación de prescripción

    # === CAMPOS VENTA LIBRE (Issue #446) ===
    # Marca y línea detectadas para productos venta libre (OTC + parafarmacia)
    detected_brand = Column(
        String(100), nullable=True, index=True
    )  # Marca detectada por BrandDetectionService (ej: "ISDIN", "BIODERMA")
    brand_line = Column(
        String(100), nullable=True
    )  # Línea de producto dentro de la marca (ej: "FOTOPROTECTOR", "HYDRABIO")

    # === CAMPOS DE CONTROL ===
    # Fuentes de datos como string separado por comas
    data_sources = Column(String(100), nullable=True)  # Fuentes de datos: 'nomenclator', 'cima', o 'nomenclator,cima'
    enrichment_confidence = Column(Integer, nullable=True)  # Confianza enriquecimiento

    # Estado de sincronización
    sync_status = Column(
        String(20), nullable=True, default="SINCRONIZADO", index=True
    )  # SINCRONIZADO, BAJA, NO_SINCRONIZADO

    # === TIMESTAMPS ===
    created_at = Column(DateTime(timezone=True), server_default=func.now(), nullable=False)
    updated_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        onupdate=func.now(),
        nullable=False,
    )

    # === ÍNDICES COMPUESTOS ===
    __table_args__ = (
        # Índices para búsquedas principales
        Index(
            "idx_nomen_principio_activo_lab",
            "nomen_principio_activo",
            "nomen_laboratorio",
        ),
        Index("idx_cima_atc_generico", "cima_atc_code", "cima_es_generico"),
        Index("idx_homogeneo_estado", "nomen_codigo_homogeneo", "nomen_estado"),
        # Índices para análisis farmacéuticos
        Index("idx_precio_referencia", "nomen_precio_referencia"),
        # Índices para datos de calidad
        Index("idx_data_sources", "data_sources"),
        Index("idx_enrichment_confidence", "enrichment_confidence"),
        # CRÍTICO: extend_existing para resolver conflictos MetaData
        {"extend_existing": True},
    )

    def __repr__(self):
        name = self.get_display_name()
        return f"<ProductCatalog CN:{self.national_code} - {name[:50]}>"

    def to_dict(self):
        """Convierte instancia a diccionario para APIs"""
        return {
            "id": str(self.id),
            "national_code": self.national_code,
            # Datos Nomenclator
            "nomen_nombre": self.nomen_nombre,
            "nomen_laboratorio": self.nomen_laboratorio,
            "nomen_codigo_laboratorio": self.nomen_codigo_laboratorio,
            "nomen_tipo_farmaco": self.nomen_tipo_farmaco,
            "nomen_principio_activo": self.nomen_principio_activo,
            "nomen_pvp": float(self.nomen_pvp) if self.nomen_pvp else None,
            "nomen_precio_referencia": (float(self.nomen_precio_referencia) if self.nomen_precio_referencia else None),
            "nomen_pvl_calculado": (float(self.nomen_pvl_calculado) if self.nomen_pvl_calculado else None),
            "nomen_codigo_homogeneo": self.nomen_codigo_homogeneo,
            "nomen_estado": self.nomen_estado,
            # Datos CIMA
            "cima_nombre_comercial": self.cima_nombre_comercial,
            "cima_atc_code": self.cima_atc_code,
            "cima_grupo_terapeutico": self.cima_grupo_terapeutico,
            "cima_forma_farmaceutica": self.cima_forma_farmaceutica,
            "cima_laboratorio_titular": self.cima_laboratorio_titular,
            "cima_requiere_receta": self.cima_requiere_receta,
            "cima_es_generico": self.cima_es_generico,
            "cima_uso_veterinario": self.cima_uso_veterinario,
            # Clasificación xFarma
            "xfarma_prescription_category": (
                self.xfarma_prescription_category.value
                if self.xfarma_prescription_category
                else None
            ),
            # Venta libre - Marca y línea (Issue #446)
            "detected_brand": self.detected_brand,
            "brand_line": self.brand_line,
            # Campos de control
            "data_sources": self.data_sources,
            "enrichment_confidence": self.enrichment_confidence,
            # Timestamps
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def get_display_name(self) -> str:
        """Obtiene el mejor nombre disponible para mostrar"""
        return self.cima_nombre_comercial or self.nomen_nombre or f"CN:{self.national_code}" or "Producto sin nombre"

    def get_laboratory(self) -> str:
        """Obtiene el mejor laboratorio disponible"""
        return self.cima_laboratorio_titular or self.nomen_laboratorio or "Laboratorio no disponible"

    def is_generic_any_source(self) -> bool:
        """Determina si es genérico según cualquier fuente"""
        return bool(self.cima_es_generico) or bool(
            self.nomen_laboratorio
            and any(
                lab in self.nomen_laboratorio.upper() for lab in ["NORMON", "CINFA", "TEVA", "SANDOZ", "RATIOPHARM"]
            )
        )

    def calculate_and_set_pvl(self, con_recargo_equivalencia: bool = True) -> bool:
        """
        Calcula y asigna el PVL desde PVP para productos ALTA con conjunto homogéneo
        usando la normativa oficial de márgenes farmacéuticos

        Condiciones para calcular PVL:
        - Producto debe tener nomen_pvp (PVP definido)
        - Estado debe ser 'ALTA'
        - Debe tener nomen_codigo_homogeneo (conjunto homogéneo)

        Args:
            con_recargo_equivalencia: Si aplicar recargo equivalencia (0.5%)

        Returns:
            bool: True si se calculó el PVL, False si no se pudo calcular
        """
        from ..external_data.nomenclator_integration import pvp_to_pvl

        # Verificar condiciones para calcular PVL
        if not self.should_calculate_pvl():
            return False

        # Calcular PVL usando la función oficial con tramos
        self.nomen_pvl_calculado = pvp_to_pvl(float(self.nomen_pvp), con_recargo_equivalencia)
        return True

    def should_calculate_pvl(self) -> bool:
        """
        Verifica si este producto cumple condiciones para calcular PVL

        Returns:
            bool: True si cumple las condiciones
        """
        return (
            self.nomen_pvp is not None
            and self.nomen_estado
            and self.nomen_estado.upper() == "ALTA"
            and self.nomen_codigo_homogeneo is not None
        )


# === AUTO-RECLASSIFICATION EVENT LISTENER (Issue #16 Gap 1) ===

# Campos críticos que disparan reclasificación automática
CRITICAL_CLASSIFICATION_FIELDS = frozenset([
    'nomen_pvp',
    'nomen_codigo_homogeneo',
    'cima_requiere_receta',
    'cima_uso_veterinario',
    'cima_estado_registro',
])


@event.listens_for(ProductCatalog, 'before_insert')
def _on_product_catalog_before_insert(mapper, connection, target):
    """
    Event listener que clasifica automáticamente un producto nuevo
    cuando se inserta en la base de datos.

    Issue #16 Gap 1: Auto-clasificación en Insert

    Note: No se pasa sesión DB al servicio para evitar conflictos con la
    transacción del event listener. La clasificación funciona sin DB para
    la mayoría de casos (usa datos del objeto en memoria).
    """
    # Import lazy para evitar circular imports
    from app.services.prescription_classification_service import PrescriptionClassificationService

    # Clasificar producto nuevo (sin sesión DB - usa datos en memoria)
    service = PrescriptionClassificationService()
    new_category = service.classify_product(target)

    # Asignar categoría (puede ser None para OTC)
    target.xfarma_prescription_category = new_category

    logger.debug(
        f"[AUTO_CLASSIFY] Producto {target.national_code} clasificado en insert: "
        f"categoria={new_category}"
    )


@event.listens_for(ProductCatalog, 'before_update')
def _on_product_catalog_before_update(mapper, connection, target):
    """
    Event listener que reclasifica automáticamente un producto cuando
    cambian campos críticos para la clasificación de prescripción.

    Issue #16 Gap 1: Auto-reclasificación Automática
    """
    state = inspect(target)

    # Detectar si algún campo crítico cambió
    changed_critical_fields = [
        attr.key for attr in state.attrs
        if attr.key in CRITICAL_CLASSIFICATION_FIELDS and attr.history.has_changes()
    ]

    if not changed_critical_fields:
        return  # No hay cambios críticos, salir

    # Import lazy para evitar circular imports
    from app.services.prescription_classification_service import PrescriptionClassificationService

    # Clasificar producto (sin sesión DB - usa datos del objeto en memoria)
    # Note: No se pasa sesión DB al servicio para evitar conflictos con la
    # transacción del event listener. La clasificación funciona sin DB para
    # la mayoría de casos (usa datos del objeto en memoria).
    service = PrescriptionClassificationService()
    new_category = service.classify_product(target)

    # Asignar nueva categoría (puede ser None para OTC)
    target.xfarma_prescription_category = new_category

    logger.debug(
        f"[AUTO_RECLASSIFY] Producto {target.national_code} reclasificado "
        f"por cambios en {changed_critical_fields}: categoria={new_category}"
    )
