# backend/app/models/prescription_reference_list.py
"""
Modelo PrescriptionReferenceList - Listados oficiales de productos de prescripción

Almacena códigos nacionales de productos clasificados manualmente en listados oficiales
(dietoterapéuticos, tiras reactivas, efectos y accesorios, etc.)

Se usa como referencia para clasificar productos automáticamente.
"""

import uuid
from datetime import datetime, timezone

from sqlalchemy import Column, DateTime, Enum as SQLEnum, Numeric, String
from sqlalchemy.dialects.postgresql import UUID

from .base import Base
from .enums import PrescriptionCategory


class PrescriptionReferenceList(Base):
    """
    Listado de referencia de productos de prescripción clasificados manualmente

    Fuente: Excel oficial con listados de:
    - Dietoterapéuticos (Dietas)
    - Tiras reactivas glucosa
    - Incontinencia financiada
    - Ortopedia financiada
    - Efectos y accesorios
    - Fórmulas magistrales
    - Vacunas hipoalergénicas

    Se actualiza manualmente cuando se publican nuevos listados oficiales.
    """

    __tablename__ = "prescription_reference_list"
    __table_args__ = {"extend_existing": True}

    # === CAMPOS BASE ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Identificador principal
    national_code = Column(
        String(20),
        nullable=False,
        index=True
    )  # Código Nacional (puede haber duplicados en categorías diferentes)

    # Información del producto
    product_name = Column(String(500), nullable=False)  # Descripción del producto
    product_description_long = Column(String(500), nullable=True)  # Descripción larga (columna 5)

    # Clasificación
    category = Column(
        SQLEnum(PrescriptionCategory, name="prescription_category_enum", values_callable=lambda x: [e.value for e in x]),
        nullable=False,
        index=True
    )  # Categoría asignada manualmente

    # Metadata de referencia
    reference_source = Column(String(100), nullable=False)  # Hoja del Excel (ej: "Dietas", "Efectos y accesorios")
    pf_code = Column(String(20), nullable=True)  # Código PF (presentación farmacéutica)

    # Información económica
    pvp_iva = Column(Numeric(10, 2), nullable=True)  # PVP con IVA
    pmf = Column(Numeric(10, 2), nullable=True)  # Precio Máximo Financiado

    # Estado
    status = Column(String(10), nullable=True)  # 'A' (Alta), 'B' (Baja)

    # Metadatos
    loaded_at = Column(
        DateTime,
        nullable=False,
        default=lambda: datetime.now(timezone.utc)
    )  # Cuándo se cargó este registro

    updated_at = Column(
        DateTime,
        nullable=True,
        default=lambda: datetime.now(timezone.utc),
        onupdate=lambda: datetime.now(timezone.utc)
    )

    def __repr__(self):
        return (
            f"<PrescriptionReferenceList("
            f"national_code='{self.national_code}', "
            f"category='{self.category.value}', "
            f"product_name='{self.product_name[:30]}...')>"
        )
