# backend/app/models/pharmacy_targets.py
"""
Modelo para gestionar targets/benchmarks configurables por farmacia.

Issue #510: Benchmarks y Líneas de Referencia en Gráficos
Permite definir líneas de referencia (mínimo, objetivo, excelente) para
contextualizar rendimiento en gráficos del dashboard VentaLibre.
"""

import uuid
from decimal import Decimal

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Numeric, String, UniqueConstraint
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from ..database import Base
from ..utils.datetime_utils import utc_now


# Valores por defecto para targets
DEFAULT_TARGETS = [
    {
        "target_type": "margin_min",
        "value": Decimal("18.0"),
        "color": "#dc3545",  # Rojo
        "line_style": "dot",
        "label": "Mínimo 18%",
    },
    {
        "target_type": "margin_target",
        "value": Decimal("25.0"),
        "color": "#28a745",  # Verde
        "line_style": "dash",
        "label": "Objetivo 25%",
    },
    {
        "target_type": "margin_excellent",
        "value": Decimal("32.0"),
        "color": "#17a2b8",  # Azul
        "line_style": "solid",
        "label": "Excelente 32%",
    },
    {
        "target_type": "hhi_max",
        "value": Decimal("2500"),
        "color": "#ffc107",  # Amarillo
        "line_style": "dash",
        "label": "HHI Máx",
    },
]


class PharmacyTarget(Base):
    """
    Modelo para gestionar targets/benchmarks configurables por farmacia.

    Permite definir líneas de referencia en gráficos del dashboard:
    - margin_min: Margen mínimo aceptable (línea roja)
    - margin_target: Margen objetivo (línea verde)
    - margin_excellent: Margen excelente (línea azul)
    - hhi_max: Concentración máxima HHI aceptable
    - sales_target: Objetivo de ventas
    - price_avg: Precio promedio de referencia

    Los targets pueden ser globales o específicos por categoría NECESIDAD.
    """

    __tablename__ = "pharmacy_targets"

    # Clave primaria
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Relación con farmacia
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # Tipo de target
    target_type = Column(
        String(50),
        nullable=False,
        index=True,
        comment="Tipo de target: margin_min, margin_target, margin_excellent, hhi_max, sales_target, price_avg",
    )

    # Categoría (opcional - NULL = global para toda la farmacia)
    category = Column(
        String(100),
        nullable=True,
        index=True,
        comment="Categoría NECESIDAD específica o NULL para target global",
    )

    # Valor del target
    value = Column(
        Numeric(12, 2),
        nullable=False,
        comment="Valor numérico del target (porcentaje, cantidad, etc.)",
    )

    # Configuración visual
    color = Column(
        String(20),
        default="#28a745",
        nullable=False,
        comment="Color de la línea en formato hex (#RRGGBB)",
    )
    line_style = Column(
        String(20),
        default="dash",
        nullable=False,
        comment="Estilo de línea: solid, dash, dot",
    )
    label = Column(
        String(100),
        nullable=True,
        comment="Etiqueta mostrada junto a la línea de referencia",
    )

    # Metadata
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now)
    user_configured = Column(
        Boolean,
        default=False,
        nullable=False,
        comment="TRUE si fue configurado manualmente por el usuario (no default)",
    )

    # Relaciones
    pharmacy = relationship("Pharmacy", back_populates="targets")

    # Constraints
    __table_args__ = (
        UniqueConstraint(
            "pharmacy_id",
            "target_type",
            "category",
            name="uq_pharmacy_target_type_category",
        ),
        {"extend_existing": True},
    )

    def to_dict(self) -> dict:
        """Convierte instancia a diccionario para serialización."""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "target_type": self.target_type,
            "category": self.category,
            "value": float(self.value) if self.value else None,
            "color": self.color,
            "line_style": self.line_style,
            "label": self.label,
            "user_configured": self.user_configured,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def to_chart_dict(self) -> dict:
        """Convierte a formato optimizado para Plotly charts."""
        return {
            "value": float(self.value) if self.value else None,
            "color": self.color,
            "line_style": self.line_style,
            "label": self.label or "",
        }

    def __repr__(self) -> str:
        return (
            f"<PharmacyTarget(pharmacy_id={self.pharmacy_id}, "
            f"type={self.target_type}, category={self.category}, value={self.value})>"
        )
