# backend/app/models/pharmacy_partners.py
"""
Modelo para gestionar laboratorios partners personalizados por farmacia
"""

import uuid

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, String, Text, UniqueConstraint
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from ..database import Base
from ..utils.datetime_utils import utc_now


class PharmacyPartner(Base):
    """
    Modelo para gestionar laboratorios partners seleccionados por cada farmacia

    Permite personalización dinámica de qué laboratorios considera cada farmacia
    como "partners" para el análisis de genéricos
    """

    __tablename__ = "pharmacy_partners"
    __table_args__ = {"extend_existing": True}

    # Clave primaria
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Relación con farmacia
    pharmacy_id = Column(UUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)

    # Información del laboratorio partner
    laboratory_name = Column(String(200), nullable=False, index=True)

    # Estado de selección
    is_selected = Column(Boolean, default=True, nullable=False)
    is_auto_suggested = Column(Boolean, default=False, nullable=False)  # Si fue sugerido automáticamente
    user_configured = Column(
        Boolean,
        default=False,
        nullable=False,
        comment="TRUE si el usuario ha configurado manualmente los partners (persiste después de reinicios)"
    )

    # Metadatos de la sugerencia automática
    suggestion_rank = Column(String(10), nullable=True)  # "1", "2", ..., "8" para top 8
    suggestion_sales_amount = Column(String(20), nullable=True)  # Ventas que justificaron la sugerencia
    suggestion_reason = Column(Text, nullable=True)  # Explicación de por qué se sugirió

    # Auditoría
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(
        DateTime(timezone=True),
        default=utc_now,
        onupdate=utc_now,
    )
    user_modified_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Timestamp de la última modificación manual del usuario (NULL = solo inicialización automática)"
    )

    # Relaciones
    pharmacy = relationship("Pharmacy", back_populates="partners")

    # Constraints
    __table_args__ = (
        UniqueConstraint("pharmacy_id", "laboratory_name", name="uq_pharmacy_laboratory"),
        {"extend_existing": True},
    )

    def to_dict(self) -> dict:
        """Convierte instancia a diccionario"""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "laboratory_name": self.laboratory_name,
            "is_selected": self.is_selected,
            "is_auto_suggested": self.is_auto_suggested,
            "user_configured": self.user_configured,
            "suggestion_rank": self.suggestion_rank,
            "suggestion_sales_amount": self.suggestion_sales_amount,
            "suggestion_reason": self.suggestion_reason,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
            "user_modified_at": self.user_modified_at.isoformat() if self.user_modified_at else None,
        }

    def __repr__(self):
        return f"<PharmacyPartner(pharmacy_id={self.pharmacy_id}, laboratory={self.laboratory_name}, selected={self.is_selected})>"
