# backend/app/models/pharmacy_homogeneous_metrics.py
"""
Modelo PharmacyHomogeneousMetrics - Métricas de conjuntos homogéneos por farmacia
Contiene métricas de ventas y configuración de partners específicas para cada farmacia
"""

import uuid

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Index, Integer, Numeric, String
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.utils.datetime_utils import utc_now

from .base import Base


class PharmacyHomogeneousMetrics(Base):
    """
    Métricas de conjuntos homogéneos específicas por farmacia.

    Esta tabla contiene:
    - Métricas de ventas por conjunto homogéneo y farmacia
    - Configuración de laboratorios partners específicos de cada farmacia
    - Cálculos de oportunidades de descuento
    - Estadísticas de rendimiento por período

    Relación: Una farmacia tiene múltiples métricas (una por conjunto homogéneo)
    Referencia: Se relaciona con HomogeneousGroupMaster para datos del catálogo
    """

    __tablename__ = "pharmacy_homogeneous_metrics"
    __table_args__ = (
        # Índices compuestos para consultas frecuentes
        Index("ix_pharmacy_metrics_combo", "pharmacy_id", "homogeneous_code"),
        Index("ix_pharmacy_metrics_partners", "pharmacy_id", "has_partner_sales"),
        Index("ix_pharmacy_metrics_opportunities", "pharmacy_id", "non_partner_units_sold"),
        Index("ix_pharmacy_metrics_updated", "updated_at"),
        {"extend_existing": True},
    )

    # === CAMPOS IDENTIFICADORES ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )
    homogeneous_code = Column(String(50), nullable=False, index=True)  # Referencia a master

    # === CONFIGURACIÓN DE PARTNERS POR FARMACIA ===
    # Cada farmacia define sus propios laboratorios partners
    partner_laboratories = Column(JSONB, nullable=True)  # Labs partners para esta farmacia: ["NORMON S.A.", "CINFA"]
    has_partner_sales = Column(Boolean, default=False, index=True)  # Tiene ventas de partners configurados

    # === ESTADÍSTICAS DE VENTAS AGREGADAS ===
    # Período de datos sobre el que se calcularon las métricas
    data_period_start = Column(DateTime, nullable=True)  # Inicio del período de datos
    data_period_end = Column(DateTime, nullable=True)  # Fin del período de datos

    # Total ventas del conjunto homogéneo en este período
    total_units_sold = Column(Integer, default=0)  # Total unidades vendidas
    total_transactions = Column(Integer, default=0)  # Total transacciones
    total_revenue = Column(Numeric(12, 2), default=0)  # Ingresos totales €

    # Ventas de laboratorios partners (configurados por la farmacia)
    partner_units_sold = Column(Integer, default=0)  # Unidades de partners
    partner_revenue = Column(Numeric(12, 2), default=0)  # Ingresos de partners €
    partner_transactions = Column(Integer, default=0)  # Transacciones con partners

    # Ventas de laboratorios NO partners (oportunidad de descuento)
    non_partner_units_sold = Column(Integer, default=0, index=True)  # CLAVE: Unidades NO-partner
    non_partner_revenue = Column(Numeric(12, 2), default=0)  # Ingresos NO-partners €
    non_partner_transactions = Column(Integer, default=0)  # Transacciones NO-partners

    # === CÁLCULOS DE OPORTUNIDADES ===
    # Base para cálculo de ahorros potenciales
    potential_savings_base = Column(Numeric(12, 2), default=0)  # Base PVL para cálculo descuento
    avg_discount_opportunity = Column(Numeric(5, 2), default=0)  # % promedio de oportunidad de descuento

    # === ANÁLISIS DE RENDIMIENTO ===
    # Información sobre el rendimiento del conjunto en esta farmacia
    best_selling_product_code = Column(String(50), nullable=True)  # Producto más vendido del conjunto
    most_profitable_lab = Column(String(200), nullable=True)  # Laboratorio más rentable

    # Comparativas con promedios del sector
    market_share_percentage = Column(Numeric(5, 2), default=0)  # % de cuota de mercado estimada
    performance_vs_average = Column(String(20), default="AVERAGE")  # ABOVE_AVERAGE, AVERAGE, BELOW_AVERAGE

    # === METADATOS Y CONTROL ===
    calculation_metadata = Column(JSONB, nullable=True)  # Metadatos del cálculo
    """
    Ejemplo calculation_metadata:
    {
        "calculation_method": "sales_aggregation",
        "data_completeness": 0.95,
        "total_sales_records": 1247,
        "partner_config_date": "2024-01-10",
        "pvl_source": "nomenclator",
        "anomalies_detected": [],
        "quality_score": 0.92
    }
    """

    # Control de actualización
    last_sales_date = Column(DateTime, nullable=True)  # Última venta incluida en cálculo

    # Timestamps de control
    created_at = Column(DateTime, default=utc_now, nullable=False)
    updated_at = Column(
        DateTime,
        default=utc_now,
        onupdate=utc_now,
        nullable=False,
        index=True,
    )

    # === RELACIONES ===
    pharmacy = relationship("Pharmacy", back_populates="homogeneous_metrics")

    def __repr__(self):
        return f"<PharmacyHomogeneousMetrics {self.homogeneous_code}: {self.non_partner_units_sold} opp. units>"

    def to_dict(self, include_metadata: bool = False) -> dict:
        """
        Convierte a diccionario para APIs de métricas por farmacia

        Args:
            include_metadata: Si incluir metadatos detallados
        """
        result = {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "homogeneous_code": self.homogeneous_code,
            # Configuración de partners
            "partner_laboratories": self.partner_laboratories or [],
            "has_partner_sales": self.has_partner_sales,
            # Métricas de ventas
            "total_units_sold": self.total_units_sold,
            "total_revenue": float(self.total_revenue) if self.total_revenue else 0.0,
            "total_transactions": self.total_transactions,
            # Métricas de partners
            "partner_units_sold": self.partner_units_sold,
            "partner_revenue": (float(self.partner_revenue) if self.partner_revenue else 0.0),
            # Oportunidades (datos clave)
            "non_partner_units_sold": self.non_partner_units_sold,
            "non_partner_revenue": (float(self.non_partner_revenue) if self.non_partner_revenue else 0.0),
            "potential_savings_base": (float(self.potential_savings_base) if self.potential_savings_base else 0.0),
            # Análisis
            "partner_penetration_percentage": self.partner_penetration_percentage,
            "avg_discount_opportunity": (
                float(self.avg_discount_opportunity) if self.avg_discount_opportunity else 0.0
            ),
            # Control temporal
            "data_period_start": (self.data_period_start.isoformat() if self.data_period_start else None),
            "data_period_end": (self.data_period_end.isoformat() if self.data_period_end else None),
            "last_sales_date": (self.last_sales_date.isoformat() if self.last_sales_date else None),
            "updated_at": self.updated_at.isoformat(),
        }

        if include_metadata:
            result.update(
                {
                    "calculation_metadata": self.calculation_metadata,
                    "best_selling_product_code": self.best_selling_product_code,
                    "most_profitable_lab": self.most_profitable_lab,
                    "market_share_percentage": (
                        float(self.market_share_percentage) if self.market_share_percentage else 0.0
                    ),
                    "performance_vs_average": self.performance_vs_average,
                }
            )

        return result

    @property
    def partner_penetration_percentage(self) -> float:
        """
        Penetración de laboratorios partners en el conjunto (0-100%)

        Returns:
            float: Porcentaje de ventas que son de partners configurados
        """
        if self.total_units_sold == 0:
            return 0.0
        return round((self.partner_units_sold / self.total_units_sold) * 100, 2)

    @property
    def opportunity_potential_score(self) -> float:
        """
        Score de potencial de oportunidad (0-100)
        Basado en unidades no-partner y base de ahorro
        """
        if not self.non_partner_units_sold or not self.potential_savings_base:
            return 0.0

        # Normalizar por volumen de unidades y base monetaria
        units_score = min(self.non_partner_units_sold / 100, 1.0) * 50  # Max 50 puntos por volumen
        savings_score = min(float(self.potential_savings_base) / 1000, 1.0) * 50  # Max 50 puntos por base €

        return round(units_score + savings_score, 2)

    def calculate_savings_with_discount(self, discount_percentage: float) -> dict:
        """
        Calcula ahorro potencial aplicando descuento sobre la base

        Args:
            discount_percentage: Descuento a aplicar (ej: 20.0 para 20%)

        Returns:
            dict: Información de ahorro calculado para esta farmacia
        """
        if not self.potential_savings_base or discount_percentage <= 0:
            return {
                "savings_amount": 0.0,
                "units_affected": self.non_partner_units_sold,
                "discount_applied": discount_percentage,
                "base_amount": float(self.potential_savings_base or 0),
                "pharmacy_id": str(self.pharmacy_id),
            }

        base_amount = float(self.potential_savings_base)
        savings = base_amount * (discount_percentage / 100)

        return {
            "savings_amount": round(savings, 2),
            "units_affected": self.non_partner_units_sold,
            "discount_applied": discount_percentage,
            "base_amount": base_amount,
            "savings_per_unit": round(savings / max(self.non_partner_units_sold, 1), 2),
            "pharmacy_id": str(self.pharmacy_id),
        }

    def get_partner_laboratories_list(self) -> list:
        """
        Obtiene lista limpia de laboratorios partners configurados para esta farmacia
        """
        if not self.partner_laboratories:
            return []

        # Asegurar que es una lista y filtrar valores nulos/vacíos
        labs = self.partner_laboratories if isinstance(self.partner_laboratories, list) else []
        return [lab for lab in labs if lab and lab.strip()]
