# backend/app/models/pharmacy.py
import uuid

from sqlalchemy import Boolean, Column, DateTime, String
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class Pharmacy(Base):
    """Modelo para farmacias en el sistema"""

    __tablename__ = "pharmacies"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Información básica
    name = Column(String(255), nullable=False)
    code = Column(String(50), unique=True, index=True)  # Código único de la farmacia
    nif = Column(String(20))  # NIF/CIF
    license_number = Column(String(50))  # Número de colegiado

    # Dirección
    address = Column(String(500))
    city = Column(String(100))
    province = Column(String(100))
    postal_code = Column(String(10))
    country = Column(String(100), default="España")

    # Códigos oficiales INE/ISO (Issue #504)
    municipality_code = Column(String(10), index=True)  # Código INE municipio (ej: 41091)
    region_code = Column(String(5), index=True)  # Código ISO 3166-2:ES (ej: ES-AN)

    # Contacto
    phone = Column(String(20))
    mobile = Column(String(20))
    email = Column(String(255))
    website = Column(String(255))

    # Configuración técnica
    erp_type = Column(String(50))  # 'farmatic', 'farmanager', 'nixfarma', etc.
    erp_version = Column(String(50))  # Versión del ERP si se conoce

    # Suscripción y estado
    subscription_plan = Column(String(50), default="trial")  # 'trial', 'basic', 'pro', 'enterprise'
    # Issue #444: Timezone-aware for consistency with User model
    subscription_start = Column(DateTime(timezone=True))
    subscription_end = Column(DateTime(timezone=True))
    is_active = Column(Boolean, default=True)
    is_demo = Column(Boolean, default=False)

    # Configuración de análisis
    analysis_period_months = Column(String(10), default="13")  # Meses de análisis por defecto
    currency = Column(String(3), default="EUR")

    # Laboratorios partners (Top 8 laboratorios genéricos por ventas)
    partner_laboratories = Column(JSONB, nullable=True)  # Lista de 8 laboratorios partners

    # Timestamps
    created_at = Column(DateTime, default=utc_now)
    updated_at = Column(DateTime, default=utc_now, onupdate=utc_now)
    last_upload_at = Column(DateTime)

    # Relaciones con otras tablas
    # Issue #348: Remover cascades SQLAlchemy, usar solo FK cascades de PostgreSQL
    user = relationship(  # ✅ Singular (relación 1:1)
        "User",
        back_populates="pharmacy",
        uselist=False,  # ✅ CRÍTICO: Forzar objeto único, no lista
        # Relación 1:1 - una farmacia tiene exactamente UN usuario
    )
    uploads = relationship("FileUpload", back_populates="pharmacy")
    sales_data = relationship("SalesData", back_populates="pharmacy")
    partners = relationship("PharmacyPartner", back_populates="pharmacy")
    homogeneous_groups = relationship("HomogeneousGroup", back_populates="pharmacy")
    homogeneous_metrics = relationship(
        "PharmacyHomogeneousMetrics",
        back_populates="pharmacy",
    )
    inventory_snapshots = relationship(
        "InventorySnapshot", back_populates="pharmacy"
    )  # Issue #476
    targets = relationship(
        "PharmacyTarget", back_populates="pharmacy", cascade="all, delete-orphan"
    )  # Issue #510: Benchmarks y líneas de referencia
    seasonality_indices = relationship(
        "SeasonalityIndex", back_populates="pharmacy", cascade="all, delete-orphan"
    )  # Issue #507: Indices de estacionalidad
    # NOTE: ERP sync state is now in ERPSyncState (single-row, no pharmacy FK)
    # Config comes from environment variables (Pivot 2026 local single-tenant)

    def __repr__(self):
        return f"<Pharmacy {self.name} ({self.code})>"

    def to_dict(self):
        """Convierte el modelo a diccionario para APIs"""
        return {
            "id": str(self.id),
            "name": self.name,
            "code": self.code,
            "nif": self.nif,
            "city": self.city,
            "province": self.province,
            "postal_code": self.postal_code,
            "municipality_code": self.municipality_code,  # Issue #504
            "region_code": self.region_code,  # Issue #504
            "email": self.email,
            "phone": self.phone,
            "erp_type": self.erp_type,
            "subscription_plan": self.subscription_plan,
            "is_active": self.is_active,
            "partner_laboratories": self.partner_laboratories or [],
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "last_upload_at": (self.last_upload_at.isoformat() if self.last_upload_at else None),
        }

    def can_upload(self) -> bool:
        """Verifica si la farmacia puede subir archivos"""
        if not self.is_active:
            return False

        # Verificar límites según plan
        plan_limits = {
            "trial": 10,  # 10 uploads en trial
            "basic": 50,
            "pro": 500,
            "enterprise": -1,  # Sin límite
        }

        limit = plan_limits.get(self.subscription_plan, 10)

        if limit == -1:
            return True

        # Contar uploads del mes actual
        # TODO: Implementar conteo real
        return True
