# backend/app/models/nomenclator_local.py
"""
Modelo NomenclatorLocal - Catálogo local del nomenclator oficial
Almacena datos oficiales del Ministerio de Sanidad descargados del CSV
"""

import uuid

from sqlalchemy import Boolean, Column, Date, DateTime, Numeric, String
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.sql import func

from .base import Base


class NomenclatorLocal(Base):
    """
    Catálogo local del nomenclator oficial del Ministerio de Sanidad

    Almacena información esencial para:
    - Identificación de laboratorios
    - Clasificación de genéricos
    - Conjuntos homogéneos
    - Precios de referencia

    Se actualiza automáticamente cada 10 días descargando el CSV oficial
    """

    __tablename__ = "nomenclator_local"
    __table_args__ = {"extend_existing": True}

    # === CAMPOS BASE ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Identificador principal
    national_code = Column(String(20), nullable=False, unique=True, index=True)

    # Información del producto
    product_name = Column(String(255), nullable=False)
    active_ingredient = Column(String(255), nullable=True)
    laboratory = Column(String(255), nullable=True, index=True)

    # Clasificación
    is_generic = Column(Boolean, nullable=True, default=False)
    requires_prescription = Column(Boolean, nullable=True, default=True)
    therapeutic_group = Column(String(100), nullable=True)

    # Precio de referencia
    reference_price = Column(Numeric(10, 2), nullable=True)

    # === CAMPOS HOMOGÉNEOS CRÍTICOS ===
    homogeneous_code = Column(String(50), nullable=True, index=True)
    homogeneous_name = Column(String(500), nullable=True)
    status = Column(String(20), nullable=True)  # 'ALTA', 'BAJA'
    pvp = Column(Numeric(10, 2), nullable=True)
    pvp_iva = Column(Numeric(10, 2), nullable=True)
    user_contribution = Column(Numeric(10, 2), nullable=True)

    # Campos adicionales útiles
    lab_code = Column(String(20), nullable=True)
    drug_type = Column(String(50), nullable=True)  # 'GENERICO', 'MARCA'
    discharge_date = Column(Date, nullable=True)
    withdrawal_date = Column(Date, nullable=True)

    # Metadatos
    updated_at = Column(DateTime, nullable=True, server_default=func.current_timestamp())

    def to_dict(self) -> dict:
        """Convierte el modelo a diccionario"""
        return {
            "id": str(self.id),
            "national_code": self.national_code,
            "product_name": self.product_name,
            "active_ingredient": self.active_ingredient,
            "laboratory": self.laboratory,
            "is_generic": self.is_generic,
            "requires_prescription": self.requires_prescription,
            "therapeutic_group": self.therapeutic_group,
            "reference_price": (float(self.reference_price) if self.reference_price else None),
            "homogeneous_code": self.homogeneous_code,
            "homogeneous_name": self.homogeneous_name,
            "status": self.status,
            "pvp": float(self.pvp) if self.pvp else None,
            "pvp_iva": float(self.pvp_iva) if self.pvp_iva else None,
            "user_contribution": (float(self.user_contribution) if self.user_contribution else None),
            "lab_code": self.lab_code,
            "drug_type": self.drug_type,
            "discharge_date": (self.discharge_date.isoformat() if self.discharge_date else None),
            "withdrawal_date": (self.withdrawal_date.isoformat() if self.withdrawal_date else None),
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def __repr__(self):
        return f"<NomenclatorLocal(national_code='{self.national_code}', product_name='{self.product_name}', laboratory='{self.laboratory}')>"
