# backend/app/models/narrative_feedback.py
"""
Modelo para almacenar feedback de usuarios sobre narrativas generadas.

Issue #509: Executive Summary con NLG (NarrativeService).

Filosofía: El feedback permite medir la efectividad del LLM y mejorar
los prompts con el tiempo. Cada feedback incluye un snapshot de los
datos usados para generar la narrativa (para debugging de alucinaciones).

Acciones:
- HELPFUL (👍): La narrativa fue útil y precisa
- NOT_HELPFUL (👎): La narrativa no fue útil o contenía errores
"""

from datetime import datetime, timezone
from uuid import uuid4

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Index, String, Text
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.database import Base


class NarrativeFeedback(Base):
    """
    Registro de feedback del usuario sobre narrativas generadas.

    El narrative_hash permite identificar la misma narrativa,
    incluso si los datos subyacentes cambian.

    Almacena snapshots de KPIs e insights para debugging
    en caso de que el LLM haya alucinado.
    """

    __tablename__ = "narrative_feedbacks"

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid4)

    # Foreign key to pharmacy (index defined in __table_args__)
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
    )

    # Narrative identification
    narrative_hash = Column(
        String(64),
        nullable=False,
        comment="SHA256[:16] del contenido de la narrativa para identificación",
    )

    # User feedback (True = 👍, False = 👎)
    is_helpful = Column(
        Boolean,
        nullable=False,
        comment="True si el usuario marcó como útil (👍), False si no (👎)",
    )

    # Optional feedback text
    feedback_text = Column(
        Text,
        nullable=True,
        comment="Texto opcional explicando el feedback del usuario",
    )

    # Snapshots para debugging de alucinaciones
    kpis_snapshot = Column(
        JSONB,
        nullable=True,
        comment="Snapshot de los KPIs usados para generar la narrativa",
    )
    insights_snapshot = Column(
        JSONB,
        nullable=True,
        comment="Snapshot de los insights del Insight Engine usados",
    )
    narrative_content = Column(
        JSONB,
        nullable=True,
        comment="Contenido completo de la narrativa generada",
    )

    # Grounding validation result at generation time
    grounding_valid = Column(
        Boolean,
        nullable=True,
        comment="True si la narrativa pasó validación de grounding",
    )

    # Timestamps
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False,
    )

    # Relationships
    pharmacy = relationship("Pharmacy", lazy="selectin")

    # Indexes for common queries
    __table_args__ = (
        # Index for looking up feedbacks by pharmacy
        Index("ix_narrative_feedback_pharmacy", "pharmacy_id"),
        # Index for finding specific narrative by hash
        Index("ix_narrative_feedback_hash", "narrative_hash"),
        # Composite for analytics: pharmacy + is_helpful
        Index(
            "ix_narrative_feedback_pharmacy_helpful",
            "pharmacy_id",
            "is_helpful",
        ),
        # Index for created_at to find recent feedbacks
        Index("ix_narrative_feedback_created_at", "created_at"),
    )

    def __repr__(self):
        return (
            f"<NarrativeFeedback(id={self.id}, "
            f"pharmacy_id={self.pharmacy_id}, "
            f"is_helpful={self.is_helpful})>"
        )

    @property
    def feedback_emoji(self) -> str:
        """Emoji representation of feedback."""
        return "👍" if self.is_helpful else "👎"
