# backend/app/models/materialized_views.py
"""
Modelos para Vistas Materializadas - Issue #543

Tablas pre-calculadas que se actualizan cada noche a las 4:00 AM
para que los dashboards carguen en <2 segundos.

Convención de nombres: mv_* (materialized view)
"""

from datetime import date, datetime
from uuid import UUID

from sqlalchemy import (
    Column,
    String,
    Integer,
    Numeric,
    Boolean,
    Date,
    DateTime,
    ForeignKey,
    Index,
    text,
)
from sqlalchemy.dialects.postgresql import UUID as PGUUID

from app.database import Base


class MVSalesMonthly(Base):
    """
    Ventas agregadas por mes y categoría.

    Granularidad: pharmacy_id + year_month + category
    Uso: Dashboard de evolución, comparativas YoY
    """
    __tablename__ = "mv_sales_monthly"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)
    year_month = Column(Date, nullable=False)  # Primer día del mes

    # Métricas de venta libre
    vl_total_sales = Column(Numeric(12, 2), default=0)
    vl_total_units = Column(Integer, default=0)
    vl_total_margin = Column(Numeric(12, 2), default=0)
    vl_margin_percent = Column(Numeric(5, 2), default=0)
    vl_sku_count = Column(Integer, default=0)

    # Métricas de prescripción
    rx_total_sales = Column(Numeric(12, 2), default=0)
    rx_total_units = Column(Integer, default=0)
    rx_total_margin = Column(Numeric(12, 2), default=0)
    rx_sku_count = Column(Integer, default=0)

    # Totales
    total_sales = Column(Numeric(12, 2), default=0)
    total_units = Column(Integer, default=0)
    total_margin = Column(Numeric(12, 2), default=0)

    # Metadata
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))

    __table_args__ = (
        Index("ix_mv_sales_monthly_pharmacy_month", "pharmacy_id", "year_month"),
        Index("ix_mv_sales_monthly_month", "year_month"),
    )


class MVSalesCategory(Base):
    """
    Ventas agregadas por categoría NECESIDAD (L1).

    Granularidad: pharmacy_id + year_month + ml_category
    Uso: Treemap de categorías, análisis por necesidad
    """
    __tablename__ = "mv_sales_category"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)
    year_month = Column(Date, nullable=False)
    ml_category = Column(String(50), nullable=False)  # L1 NECESIDAD

    # Métricas
    total_sales = Column(Numeric(12, 2), default=0)
    total_units = Column(Integer, default=0)
    total_margin = Column(Numeric(12, 2), default=0)
    margin_percent = Column(Numeric(5, 2), default=0)
    sku_count = Column(Integer, default=0)
    transaction_count = Column(Integer, default=0)

    # Top marca en la categoría
    top_brand = Column(String(100), nullable=True)
    top_brand_sales = Column(Numeric(12, 2), default=0)

    # Metadata
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))

    __table_args__ = (
        Index("ix_mv_sales_category_pharmacy_month_cat", "pharmacy_id", "year_month", "ml_category"),
    )


class MVSalesBrand(Base):
    """
    Ventas agregadas por marca.

    Granularidad: pharmacy_id + year_month + detected_brand
    Uso: Análisis de marcas, brand duel, competencia
    """
    __tablename__ = "mv_sales_brand"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)
    year_month = Column(Date, nullable=False)
    detected_brand = Column(String(100), nullable=False)
    ml_category = Column(String(50), nullable=True)  # Categoría principal de la marca

    # Métricas
    total_sales = Column(Numeric(12, 2), default=0)
    total_units = Column(Integer, default=0)
    total_margin = Column(Numeric(12, 2), default=0)
    margin_percent = Column(Numeric(5, 2), default=0)
    sku_count = Column(Integer, default=0)

    # Metadata
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))

    __table_args__ = (
        Index("ix_mv_sales_brand_pharmacy_month_brand", "pharmacy_id", "year_month", "detected_brand"),
    )


class MVStockKPIs(Base):
    """
    KPIs de stock por producto.

    Granularidad: pharmacy_id + product_code
    Uso: Alertas de stock muerto, reposición, liquidación
    """
    __tablename__ = "mv_stock_kpis"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)
    product_code = Column(String(20), nullable=False)  # CN o código interno
    product_name = Column(String(255), nullable=True)

    # Stock actual
    stock_qty = Column(Integer, default=0)
    stock_value = Column(Numeric(12, 2), default=0)

    # Velocidad de venta (últimos 30 días)
    daily_velocity = Column(Numeric(8, 3), default=0)  # Unidades/día
    coverage_days = Column(Integer, default=0)  # Días de cobertura

    # Última venta
    last_sale_date = Column(Date, nullable=True)
    days_without_sale = Column(Integer, default=0)

    # Clasificación
    is_dead_stock = Column(Boolean, default=False)  # >180 días sin venta
    is_slow_mover = Column(Boolean, default=False)  # 90-180 días sin venta
    is_critical = Column(Boolean, default=False)  # <3 días cobertura

    # ABC Classification
    abc_class = Column(String(1), nullable=True)  # A, B, C

    # Metadata
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))

    __table_args__ = (
        Index("ix_mv_stock_kpis_pharmacy_product", "pharmacy_id", "product_code"),
        Index("ix_mv_stock_kpis_dead_stock", "pharmacy_id", "is_dead_stock"),
        Index("ix_mv_stock_kpis_critical", "pharmacy_id", "is_critical"),
    )


class MVWeeklyKPIs(Base):
    """
    KPIs semanales agregados para PDF Semanal.

    Granularidad: pharmacy_id + week_start
    Uso: PDF Semanal, tendencias corto plazo
    """
    __tablename__ = "mv_weekly_kpis"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=False)
    week_start = Column(Date, nullable=False)  # Lunes de la semana

    # Ventas
    total_sales = Column(Numeric(12, 2), default=0)
    vl_sales = Column(Numeric(12, 2), default=0)
    rx_sales = Column(Numeric(12, 2), default=0)

    # Margen
    total_margin = Column(Numeric(12, 2), default=0)
    margin_percent = Column(Numeric(5, 2), default=0)

    # Actividad
    transaction_count = Column(Integer, default=0)
    unique_products = Column(Integer, default=0)

    # Comparativa YoY (calculada)
    yoy_sales_change = Column(Numeric(6, 2), nullable=True)  # % cambio
    yoy_margin_change = Column(Numeric(6, 2), nullable=True)

    # Top categoría de la semana
    top_category = Column(String(50), nullable=True)
    top_category_sales = Column(Numeric(12, 2), default=0)

    # Alertas
    alert_count = Column(Integer, default=0)
    critical_alert_count = Column(Integer, default=0)

    # Metadata
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))

    __table_args__ = (
        Index("ix_mv_weekly_kpis_pharmacy_week", "pharmacy_id", "week_start"),
    )


class ETLRunLog(Base):
    """
    Log de ejecuciones del ETL nocturno.

    Uso: Auditoría, debugging, monitoreo de salud
    """
    __tablename__ = "etl_run_log"

    id = Column(Integer, primary_key=True, autoincrement=True)
    pharmacy_id = Column(PGUUID(as_uuid=True), ForeignKey("pharmacies.id"), nullable=True)

    # Ejecución
    run_type = Column(String(50), nullable=False)  # 'nightly', 'manual', 'weekly_pdf'
    started_at = Column(DateTime, nullable=False)
    completed_at = Column(DateTime, nullable=True)
    duration_seconds = Column(Integer, nullable=True)

    # Resultado
    status = Column(String(20), nullable=False)  # 'running', 'success', 'failed'
    error_message = Column(String(500), nullable=True)

    # Métricas
    rows_processed = Column(Integer, default=0)
    tables_updated = Column(Integer, default=0)

    # Detalle
    details = Column(String(2000), nullable=True)  # JSON con métricas por tabla

    __table_args__ = (
        Index("ix_etl_run_log_pharmacy_started", "pharmacy_id", "started_at"),
        Index("ix_etl_run_log_status", "status"),
    )
