# backend/app/models/license_cache.py
"""
License Cache Model for kaiFarma Local.

Pivot 2026: License validation against Hub Central with offline support.

This single-row model caches the license state locally to:
- Enable offline operation during Hub downtime
- Provide 7-day grace period before kill-switch
- Store pharmacy info for UI display
- Prevent clock manipulation (time traveler protection)
- Prevent DB cloning (machine binding)

Flow:
1. Startup: Validate against Hub -> Update cache
2. Online: Normal operation, cache refreshed every 24h
3. Offline < 7 days: Warning banner, full operation
4. Offline > 7 days: Kill-switch, read-only mode

Security:
- Time travel protection: last_validated_system_time tracks monotonic time
- Machine binding: machine_id prevents copying DB to another computer
"""
import logging
import platform
from datetime import datetime, timedelta
from typing import Optional

from sqlalchemy import Boolean, Column, DateTime, Integer, String, Text

from app.database import Base
from app.utils.datetime_utils import utc_now


logger = logging.getLogger(__name__)

# Grace period in days before kill-switch activates
GRACE_PERIOD_DAYS = 7


def get_machine_id() -> str:
    """
    Get a unique identifier for this machine.

    On Windows: Uses machine GUID from registry (hardware-bound)
    On other platforms: Uses MAC address + hostname (uuid.getnode())

    SECURITY: This identifier is used to prevent database cloning.
    The fallback is intentionally restrictive - if we can't get a
    reliable machine ID, we use a combination that's harder to spoof.

    Returns:
        String identifier for this machine

    Raises:
        RuntimeError: If machine ID cannot be determined (critical security failure)
    """
    try:
        if platform.system() == "Windows":
            import winreg
            key = winreg.OpenKey(
                winreg.HKEY_LOCAL_MACHINE,
                r"SOFTWARE\Microsoft\Cryptography",
                0,
                winreg.KEY_READ | winreg.KEY_WOW64_64KEY
            )
            machine_guid, _ = winreg.QueryValueEx(key, "MachineGuid")
            winreg.CloseKey(key)
            return f"win:{machine_guid}"
        else:
            # Non-Windows: Use MAC address (uuid.getnode) + hostname
            # This is harder to spoof than hostname alone
            import uuid
            mac_address = uuid.getnode()
            hostname = platform.node()
            return f"unix:{mac_address}:{hostname}"
    except Exception as e:
        logger.error(f"Could not get primary machine ID: {e}")
        # Fallback: Try to get MAC address at minimum
        try:
            import uuid
            mac_address = uuid.getnode()
            # uuid.getnode() returns a random if MAC not available
            # Check if it's a "fake" MAC (bit 0 of first octet is 1)
            if (mac_address >> 40) & 1:
                logger.critical(
                    "SECURITY: Cannot get real MAC address for machine binding. "
                    "This installation may be vulnerable to database cloning."
                )
            return f"mac:{mac_address}"
        except Exception as fallback_error:
            logger.critical(
                f"SECURITY CRITICAL: Cannot determine machine ID: {fallback_error}. "
                "License validation cannot proceed safely."
            )
            # Return a unique-per-session ID that will force revalidation each restart
            # This prevents using a cloned database but requires online validation
            import secrets
            return f"temp:{secrets.token_hex(16)}"


class LicenseCache(Base):
    """
    License Cache for local installation.

    Single-row table (id=1 always) caching license validation state.
    Allows offline operation with grace period.

    Environment Variables:
        LICENSE_KEY: License key (KAI-XXXX-YYYY-ZZZZ)
        HUB_URL: Hub Central URL (https://api.kaifarma.es)

    Usage:
        # Get or create the singleton cache
        cache = db.query(LicenseCache).first()
        if not cache:
            cache = LicenseCache(id=1, license_key=settings.LICENSE_KEY)
            db.add(cache)

        # Check license status
        if cache.is_kill_switch_active():
            # Enter read-only mode
            pass
        elif cache.is_in_grace_period():
            # Show warning banner
            pass
    """

    __tablename__ = "license_cache"
    __table_args__ = {"extend_existing": True}

    # Single-row table: id is always 1
    id = Column(Integer, primary_key=True, default=1)

    # License identification
    license_key = Column(String(50), nullable=False)  # KAI-XXXX-YYYY-ZZZZ

    # Pharmacy info from Hub (cached for offline display)
    pharmacy_id = Column(String(36), nullable=True)  # UUID from Hub
    pharmacy_name = Column(String(200), nullable=True)
    tier = Column(String(20), default="standard")  # free, standard, premium

    # Validation timestamps
    validated_at = Column(DateTime(timezone=True), nullable=True)  # Last validation attempt
    expires_at = Column(DateTime(timezone=True), nullable=True)  # License expiration (if any)

    # Grace period tracking (critical for offline support)
    last_online_validation = Column(DateTime(timezone=True), nullable=True)
    grace_period_ends = Column(DateTime(timezone=True), nullable=True)

    # Status
    is_valid = Column(Boolean, default=False)
    validation_error = Column(String(500), nullable=True)

    # === SECURITY: Time Travel Protection ===
    # Tracks the system time at last validation to detect clock rollback
    last_validated_system_time = Column(DateTime(timezone=True), nullable=True)

    # === SECURITY: Machine Binding (Anti-Clone) ===
    # Unique identifier for this machine to prevent DB cloning
    machine_id = Column(String(100), nullable=True)

    # Metadata
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now)

    def __repr__(self):
        return f"<LicenseCache key={self.license_key[:12]}... valid={self.is_valid} pharmacy={self.pharmacy_name}>"

    def to_dict(self) -> dict:
        """Convert to dictionary for API responses."""
        return {
            "license_key": self.license_key[:12] + "..." if self.license_key else None,
            "pharmacy_id": self.pharmacy_id,
            "pharmacy_name": self.pharmacy_name,
            "tier": self.tier,
            "is_valid": self.is_valid,
            "validated_at": self.validated_at.isoformat() if self.validated_at else None,
            "expires_at": self.expires_at.isoformat() if self.expires_at else None,
            "last_online_validation": (
                self.last_online_validation.isoformat()
                if self.last_online_validation
                else None
            ),
            "grace_period_ends": (
                self.grace_period_ends.isoformat()
                if self.grace_period_ends
                else None
            ),
            "in_grace_period": self.is_in_grace_period(),
            "kill_switch_active": self.is_kill_switch_active(),
            "days_remaining": self.days_until_kill_switch(),
            "validation_error": self.validation_error,
            # Security status (don't expose machine_id)
            "security_status": self.get_security_status(),
        }

    def update_from_hub_response(
        self,
        is_valid: bool,
        pharmacy_id: Optional[str] = None,
        pharmacy_name: Optional[str] = None,
        tier: Optional[str] = None,
        expires_at: Optional[datetime] = None,
        error: Optional[str] = None,
    ) -> None:
        """
        Update cache from Hub validation response.

        Args:
            is_valid: Whether license is valid
            pharmacy_id: Pharmacy UUID from Hub
            pharmacy_name: Pharmacy name for display
            tier: License tier (free, standard, premium)
            expires_at: License expiration date
            error: Error message if validation failed
        """
        now = utc_now()
        self.validated_at = now
        self.is_valid = is_valid

        # Security: Always update system time for time travel detection
        self.last_validated_system_time = now

        # Security: Bind to this machine on first validation
        current_machine = get_machine_id()
        if not self.machine_id:
            self.machine_id = current_machine
            logger.info(f"License bound to machine: {current_machine[:20]}...")

        if is_valid:
            # Online validation succeeded
            self.last_online_validation = now
            self.grace_period_ends = now + timedelta(days=GRACE_PERIOD_DAYS)
            self.validation_error = None

            if pharmacy_id:
                self.pharmacy_id = pharmacy_id
            if pharmacy_name:
                self.pharmacy_name = pharmacy_name
            if tier:
                self.tier = tier
            if expires_at:
                self.expires_at = expires_at
        else:
            # Validation failed - keep grace period from last successful
            self.validation_error = error

    def mark_offline_validation(self) -> None:
        """Mark that we tried to validate but Hub was unreachable."""
        self.validated_at = utc_now()
        # Don't update last_online_validation or grace_period_ends
        # They stay as they were from last successful online validation

    def is_in_grace_period(self) -> bool:
        """
        Check if we're in offline grace period.

        Returns:
            True if offline but within grace period (show warning)
        """
        if not self.last_online_validation:
            # Never validated online - no grace period
            return False

        if not self.grace_period_ends:
            return False

        now = utc_now()
        return now < self.grace_period_ends

    def is_kill_switch_active(self) -> bool:
        """
        Check if grace period has expired (kill-switch).

        Returns:
            True if offline and grace period exceeded (read-only mode)
        """
        if not self.last_online_validation:
            # Never validated online - allow first-time setup
            return False

        if not self.grace_period_ends:
            return False

        now = utc_now()
        return now >= self.grace_period_ends

    def days_until_kill_switch(self) -> Optional[int]:
        """
        Calculate days remaining until kill-switch.

        Returns:
            Days remaining (None if not in grace period)
        """
        if not self.grace_period_ends:
            return None

        now = utc_now()
        if now >= self.grace_period_ends:
            return 0

        delta = self.grace_period_ends - now
        return delta.days

    def needs_revalidation(self, interval_hours: int = 24) -> bool:
        """
        Check if license should be revalidated.

        Args:
            interval_hours: Hours between revalidations

        Returns:
            True if enough time has passed since last validation
        """
        if not self.last_online_validation:
            return True

        hours_since = (utc_now() - self.last_online_validation).total_seconds() / 3600
        return hours_since >= interval_hours

    # === SECURITY METHODS ===

    def is_time_travel_detected(self) -> bool:
        """
        Detect if system clock has been rolled back (time travel).

        If current time is BEFORE last validated time, someone has
        manipulated the system clock to extend the grace period.

        Returns:
            True if clock rollback detected (require immediate revalidation)
        """
        if not self.last_validated_system_time:
            return False

        now = utc_now()
        # Allow small margin (10 seconds) for clock sync jitter only
        # SECURITY: Larger margins allow attackers to extend grace period
        margin = timedelta(seconds=10)

        if now < (self.last_validated_system_time - margin):
            logger.critical(
                f"SECURITY: Time travel detected! System time {now} < "
                f"last validated {self.last_validated_system_time}. "
                f"Immediate online revalidation required."
            )
            return True

        return False

    def is_machine_mismatch(self) -> bool:
        """
        Detect if database was copied to a different machine.

        Compares stored machine_id with current machine's ID.

        Returns:
            True if running on different machine (require immediate revalidation)
        """
        if not self.machine_id:
            # No machine bound yet - first run
            return False

        current_machine = get_machine_id()
        if self.machine_id != current_machine:
            logger.warning(
                f"Machine mismatch! Cache bound to {self.machine_id[:20]}... "
                f"but running on {current_machine[:20]}..."
            )
            return True

        return False

    def get_security_status(self) -> str:
        """
        Get security status for API response.

        Returns:
            Status string: "ok", "time_travel", "machine_mismatch"
        """
        if self.is_time_travel_detected():
            return "time_travel"
        if self.is_machine_mismatch():
            return "machine_mismatch"
        return "ok"

    def requires_immediate_revalidation(self) -> bool:
        """
        Check if license requires IMMEDIATE online revalidation.

        This is triggered by security violations (time travel, machine mismatch).
        Grace period does NOT apply in these cases.

        Returns:
            True if security violation detected
        """
        return self.is_time_travel_detected() or self.is_machine_mismatch()
