# backend/app/models/keyword_override.py
"""
Modelo para almacenar overrides de keywords del clasificador NECESIDAD.

Issue #449: Keywords dinámicos gestionables desde UI admin.

Arquitectura de prioridad:
    1. DB Blacklist (keyword_type='blacklist') - Máxima prioridad
    2. DB Brands (keyword_type='brand') - Alta prioridad
    3. DB Keywords (keyword_type='keyword')
    4. Código MARCA_NECESIDAD - Fallback brands
    5. Código NECESIDAD_KEYWORDS - Fallback keywords
"""

import enum
from datetime import datetime, timezone

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    ForeignKey,
    Index,
    Integer,
    String,
    Text,
    UniqueConstraint,
)
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base


class KeywordType(str, enum.Enum):
    """Tipo de keyword para clasificación."""
    KEYWORD = "keyword"       # Keyword genérico (NECESIDAD_KEYWORDS)
    BRAND = "brand"           # Marca (MARCA_NECESIDAD, mayor prioridad)
    BLACKLIST = "blacklist"   # Exclusión (interno_no_venta, packs, etc.)


class KeywordOverride(Base):
    """
    Override dinámico de keyword para clasificador NECESIDAD.

    Prioridad de clasificación:
    1. DB overrides activos (is_active=True)
    2. Fallback a código Python (NECESIDAD_KEYWORDS, MARCA_NECESIDAD)

    Campos:
    - keyword: El término a buscar (ej: "olistic", "anticaida")
    - category: La categoría NECESIDAD destino (ej: "caida_cabello")
    - keyword_type: BRAND > KEYWORD (prioridad)
    - priority: Peso dentro del tipo (mayor = primero)
    - is_active: Solo activos se usan en clasificación
    """
    __tablename__ = "keyword_overrides"

    id = Column(Integer, primary_key=True, index=True)

    # === KEYWORD DATA ===
    # Almacenado en lowercase para matching case-insensitive
    keyword = Column(
        String(200),
        nullable=False,
        comment="Término a buscar (lowercase)"
    )

    # Categoría NECESIDAD destino (ej: caida_cabello, proteccion_solar)
    category = Column(
        String(100),
        nullable=False,
        comment="Categoría NECESIDAD destino"
    )

    # Tipo: keyword, brand, blacklist (stored as String for compatibility)
    keyword_type = Column(
        String(20),
        nullable=False,
        default="keyword",
        comment="Tipo: keyword, brand, blacklist"
    )

    # === PRIORITY & STATUS ===
    # Mayor prioridad = se evalúa primero
    priority = Column(
        Integer,
        nullable=False,
        default=100,
        comment="Prioridad (mayor = primero)"
    )

    is_active = Column(
        Boolean,
        nullable=False,
        default=True,
        comment="Solo activos se usan en clasificación"
    )

    # === AUDIT ===
    created_by = Column(
        UUID(as_uuid=True),
        ForeignKey("users.id", ondelete="SET NULL"),
        nullable=True,
        comment="Usuario que creó el override"
    )

    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False
    )

    updated_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        onupdate=lambda: datetime.now(timezone.utc),
        nullable=False
    )

    # === METADATA ===
    notes = Column(
        Text,
        nullable=True,
        comment="Notas del admin"
    )

    # Fuente: manual, imported, generated
    source = Column(
        String(50),
        default="manual",
        comment="Fuente: manual, imported, generated"
    )

    # === RELATIONSHIPS ===
    creator = relationship("User", lazy="selectin")

    # === INDEXES ===
    __table_args__ = (
        # Unique constraint: one keyword per type
        UniqueConstraint('keyword', 'keyword_type', name='uq_keyword_type'),
        # Performance indexes
        Index('ix_keyword_override_active', 'is_active', 'keyword_type'),
        Index('ix_keyword_override_category', 'category'),
        # Note: lowercase index created in migration
    )

    def __repr__(self):
        return (
            f"<KeywordOverride(id={self.id}, "
            f"keyword='{self.keyword}', "
            f"category='{self.category}', "
            f"type={self.keyword_type})>"
        )

    @property
    def keyword_type_enum(self) -> KeywordType:
        """Get keyword_type as enum."""
        return KeywordType(self.keyword_type)

    @property
    def is_brand(self) -> bool:
        """True if this is a brand override (higher priority)."""
        return self.keyword_type == KeywordType.BRAND.value

    @property
    def is_blacklist(self) -> bool:
        """True if this is a blacklist override (highest priority)."""
        return self.keyword_type == KeywordType.BLACKLIST.value
