# backend/app/models/invitation.py
"""
Modelo para el sistema de invitaciones de xFarma
Permite a administradores y gerentes invitar nuevos usuarios
"""

import uuid
from datetime import timedelta

from sqlalchemy import Column, DateTime, ForeignKey, String, Text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class Invitation(Base):
    """Modelo para invitaciones de usuario"""

    __tablename__ = "invitations"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Foreign keys
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
    )
    invited_by_id = Column(
        UUID(as_uuid=True),
        ForeignKey("users.id", ondelete="SET NULL"),
        nullable=True,
    )

    # Información de la invitación
    email = Column(String(255), nullable=False, index=True)
    role = Column(String(50), default="user")  # Rol asignado al nuevo usuario
    token = Column(String(255), unique=True, nullable=False, index=True)

    # Información adicional
    message = Column(Text)  # Mensaje personalizado del invitador
    full_name = Column(String(255))  # Nombre sugerido para el usuario
    dni_nie = Column(String(20))  # DNI/NIE pre-configurado

    # Estado de la invitación
    status = Column(String(20), default="pending")  # 'pending', 'accepted', 'expired', 'cancelled'
    accepted_at = Column(DateTime)
    accepted_by_id = Column(
        UUID(as_uuid=True),
        ForeignKey("users.id", ondelete="SET NULL"),
        nullable=True,
    )

    # Control de expiración
    expires_at = Column(DateTime)
    max_uses = Column(String(10), default="1")  # Número de usos permitidos (para invitaciones grupales)
    times_used = Column(String(10), default="0")

    # Timestamps
    created_at = Column(DateTime, default=utc_now)
    updated_at = Column(DateTime, default=utc_now, onupdate=utc_now)

    # Relaciones
    pharmacy = relationship("Pharmacy")
    invited_by = relationship("User", foreign_keys=[invited_by_id])
    accepted_by = relationship("User", foreign_keys=[accepted_by_id])

    def __init__(self, **kwargs):
        """Inicializa la invitación con valores por defecto"""
        super().__init__(**kwargs)

        # Generar token si no se proporciona
        if not self.token:
            self.token = str(uuid.uuid4())

        # Establecer expiración por defecto (7 días)
        if not self.expires_at:
            self.expires_at = utc_now() + timedelta(days=7)

    def __repr__(self):
        return f"<Invitation {self.email} - {self.status}>"

    def is_valid(self) -> bool:
        """Verifica si la invitación es válida"""
        if self.status != "pending":
            return False

        if utc_now() > self.expires_at:
            self.status = "expired"
            return False

        try:
            times_used = int(self.times_used or "0")
            max_uses = int(self.max_uses or "1")
            if times_used >= max_uses:
                return False
        except ValueError:
            return False

        return True

    def accept(self, user_id: UUID):
        """Marca la invitación como aceptada"""
        self.status = "accepted"
        self.accepted_at = utc_now()
        self.accepted_by_id = user_id

        # Incrementar contador de usos
        try:
            times_used = int(self.times_used or "0")
            self.times_used = str(times_used + 1)
        except ValueError:
            self.times_used = "1"

    def cancel(self):
        """Cancela la invitación"""
        self.status = "cancelled"

    def to_dict(self):
        """Convierte el modelo a diccionario para APIs"""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "email": self.email,
            "role": self.role,
            "full_name": self.full_name,
            "status": self.status,
            "expires_at": self.expires_at.isoformat() if self.expires_at else None,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "message": self.message,
        }
