# backend/app/models/inventory_snapshot.py
"""
Modelo para snapshots de inventario de farmacia.

Issue #476: Carga de ficheros de inventario desde ERPs.
Cada snapshot representa el estado del inventario en una fecha específica.
"""

import uuid
from datetime import date

from sqlalchemy import (
    Column,
    Date,
    DateTime,
    ForeignKey,
    Index,
    Integer,
    Numeric,
    String,
)
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class InventorySnapshot(Base):
    """
    Snapshot de inventario de una farmacia en una fecha específica.

    Cada registro representa un producto en el inventario.
    Se permite un snapshot por pharmacy_id + snapshot_date.
    Al subir un nuevo fichero con la misma fecha, se reemplazan los datos.
    """

    __tablename__ = "inventory_snapshots"
    __table_args__ = (
        # Índice compuesto para consultas por farmacia y fecha
        Index("idx_inventory_pharmacy_date", "pharmacy_id", "snapshot_date"),
        # Índice para búsqueda por código
        Index("idx_inventory_product_code", "product_code"),
        # Índice para búsqueda por EAN
        Index("idx_inventory_ean13", "ean13"),
        {"extend_existing": True},
    )

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Foreign keys
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
        comment="Farmacia propietaria del inventario",
    )
    file_upload_id = Column(
        UUID(as_uuid=True),
        ForeignKey("file_uploads.id", ondelete="CASCADE"),
        nullable=False,
        comment="Fichero de origen del snapshot",
    )

    # Catalog links (Issue #477: clasificación de productos)
    product_catalog_id = Column(
        UUID(as_uuid=True),
        ForeignKey("product_catalog.id", ondelete="SET NULL"),
        nullable=True,
        index=True,
        comment="Referencia al producto en catálogo CIMA/prescripción",
    )
    venta_libre_product_id = Column(
        UUID(as_uuid=True),
        ForeignKey("product_catalog_venta_libre.id", ondelete="SET NULL"),
        nullable=True,
        index=True,
        comment="Referencia al producto en catálogo VentaLibre",
    )

    # Identificación del producto
    product_code = Column(
        String(20),
        nullable=True,
        comment="Código Nacional (CN) - 6-7 dígitos",
    )
    product_name = Column(
        String(500),
        nullable=False,
        comment="Nombre del producto según ERP",
    )
    ean13 = Column(
        String(20),
        nullable=True,
        comment="Código EAN-13 completo",
    )

    # Clasificación del producto (Issue #476)
    product_type = Column(
        String(20),
        nullable=True,
        index=True,
        comment="Tipo: 'prescription', 'venta_libre', 'unknown'",
    )

    # Stock
    stock_quantity = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Unidades en stock",
    )
    stock_value = Column(
        Numeric(12, 2),
        nullable=True,
        comment="Valor del stock a PVP (unit_price * stock_quantity)",
    )
    cost_value = Column(
        Numeric(12, 2),
        nullable=True,
        comment="Valor del stock a coste (unit_cost * stock_quantity)",
    )

    # Precios unitarios
    unit_price = Column(
        Numeric(10, 2),
        nullable=True,
        comment="Precio de venta al público (PVP)",
    )
    unit_cost = Column(
        Numeric(10, 2),
        nullable=True,
        comment="Precio de coste/compra (PCM)",
    )

    # Fechas de actividad
    last_sale_date = Column(
        Date,
        nullable=True,
        comment="Fecha de última venta",
    )
    last_purchase_date = Column(
        Date,
        nullable=True,
        comment="Fecha de última compra",
    )

    # Fecha del snapshot
    snapshot_date = Column(
        Date,
        nullable=False,
        default=date.today,
        comment="Fecha del inventario (fecha del fichero)",
    )

    # Metadata
    created_at = Column(
        DateTime,
        default=utc_now,
        comment="Timestamp de creación del registro",
    )

    # Relationships
    pharmacy = relationship(
        "Pharmacy",
        back_populates="inventory_snapshots",
        lazy="joined",
    )
    file_upload = relationship(
        "FileUpload",
        back_populates="inventory_snapshots",
        lazy="joined",
    )

    # Catalog relationships (Issue #477)
    product_catalog = relationship(
        "ProductCatalog",
        foreign_keys=[product_catalog_id],
        lazy="select",
    )
    venta_libre_product = relationship(
        "ProductCatalogVentaLibre",
        foreign_keys=[venta_libre_product_id],
        lazy="select",
    )

    def __repr__(self):
        name = (self.product_name or "Unknown")[:30]
        return f"<InventorySnapshot {name}... qty={self.stock_quantity}>"

    def to_dict(self):
        """Serializar a diccionario para API responses."""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "file_upload_id": str(self.file_upload_id),
            "product_catalog_id": str(self.product_catalog_id) if self.product_catalog_id else None,
            "venta_libre_product_id": str(self.venta_libre_product_id) if self.venta_libre_product_id else None,
            "product_code": self.product_code,
            "product_name": self.product_name,
            "ean13": self.ean13,
            "product_type": self.product_type,
            "stock_quantity": self.stock_quantity,
            "stock_value": float(self.stock_value) if self.stock_value else None,
            "cost_value": float(self.cost_value) if self.cost_value else None,
            "unit_price": float(self.unit_price) if self.unit_price else None,
            "unit_cost": float(self.unit_cost) if self.unit_cost else None,
            "last_sale_date": self.last_sale_date.isoformat() if self.last_sale_date else None,
            "last_purchase_date": self.last_purchase_date.isoformat() if self.last_purchase_date else None,
            "snapshot_date": self.snapshot_date.isoformat() if self.snapshot_date else None,
            "created_at": self.created_at.isoformat() if self.created_at else None,
        }

    @classmethod
    def extract_cn_from_ean(cls, ean13: str) -> str | None:
        """
        Extrae el código nacional (CN) de un EAN-13 español.

        Formato EAN español farmacia: 847000XXXXXX (check digit)
        El CN son los 6 dígitos después de 847000.

        Args:
            ean13: Código EAN-13 completo

        Returns:
            Código nacional de 6 dígitos o None si no es EAN español
        """
        if not ean13 or len(ean13) < 12:
            return None

        # EAN español de farmacia: empieza por 847000
        if ean13.startswith("847000"):
            return ean13[6:12]

        return None

    def calculate_stock_values(self):
        """Calcula los valores de stock basados en precios y cantidad."""
        if self.unit_price and self.stock_quantity:
            self.stock_value = self.unit_price * self.stock_quantity
        if self.unit_cost and self.stock_quantity:
            self.cost_value = self.unit_cost * self.stock_quantity
