# backend/app/models/interchangeable_curated_group.py
"""
Modelo InterchangeableCuratedGroup - Grupos curados manualmente (ADR-004).

Issue #521: Admin UI para gestión de grupos intercambiables curados.
ADR-004: Simplificación del Sistema de Clasificación

Este modelo reemplaza el clustering dinámico por grupos curados manualmente:
- Los grupos se crean/editan desde el Admin UI
- Cada grupo tiene una lista de productos miembros
- Los 8 grupos de Fase 3 se migran como seed inicial
"""

import uuid
from datetime import datetime, timezone
from typing import TYPE_CHECKING, List, Optional

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Integer, Numeric, String, Text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from .base import Base


class InterchangeableCuratedGroup(Base):
    """
    Grupos intercambiables curados manualmente.

    A diferencia de IntercambiableGroup (clustering dinámico), estos grupos
    son creados y mantenidos manualmente desde el Admin UI.

    Atributos:
        group_name: Nombre descriptivo del grupo
        group_slug: Identificador URL-friendly único
        description: Descripción detallada
        necesidad_l1: Categoría L1 (NECESIDAD)
        subcategory_l2: Subcategoría L2 opcional
        source: 'legacy_clustering' o 'manual_curated'
        is_active: Si el grupo está activo
        product_count: Número de productos (denormalizado)
        brand_count: Número de marcas (denormalizado)
        total_sales_amount: Ventas totales (denormalizado)
    """

    __tablename__ = "interchangeable_curated_groups"

    # === CLAVE PRIMARIA ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # === IDENTIFICACIÓN ===
    group_name = Column(
        String(100),
        nullable=False,
        comment="Nombre descriptivo del grupo"
    )
    group_slug = Column(
        String(100),
        unique=True,
        nullable=False,
        index=True,
        comment="Identificador URL-friendly único"
    )
    description = Column(
        Text,
        nullable=True,
        comment="Descripción detallada del grupo"
    )

    # === CLASIFICACIÓN ===
    necesidad_l1 = Column(
        String(50),
        nullable=True,
        index=True,
        comment="Categoría L1 (NECESIDAD) para este grupo"
    )
    subcategory_l2 = Column(
        String(50),
        nullable=True,
        comment="Subcategoría L2 si aplica"
    )

    # === METADATA ===
    source = Column(
        String(30),
        nullable=False,
        default="manual_curated",
        comment="Origen: 'legacy_clustering' o 'manual_curated'"
    )
    is_active = Column(
        Boolean,
        nullable=False,
        default=True,
        comment="Si el grupo está activo"
    )

    # === ESTADÍSTICAS (DENORMALIZADAS) ===
    product_count = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Número de productos en el grupo"
    )
    brand_count = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Número de marcas diferentes"
    )
    total_sales_amount = Column(
        Numeric(12, 2),
        nullable=True,
        comment="Ventas totales de productos del grupo"
    )

    # === TIMESTAMPS ===
    created_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        nullable=False
    )
    updated_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        onupdate=func.now(),
        nullable=False
    )
    created_by = Column(
        String(100),
        nullable=True,
        comment="Usuario que creó/curó el grupo"
    )

    # === RELACIONES ===
    members = relationship(
        "InterchangeableGroupMember",
        back_populates="group",
        cascade="all, delete-orphan",
        lazy="dynamic"
    )

    # === CONFIGURACIÓN DE TABLA ===
    __table_args__ = (
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        return (
            f"<InterchangeableCuratedGroup("
            f"id={self.id}, "
            f"name='{self.group_name}', "
            f"l1='{self.necesidad_l1}', "
            f"products={self.product_count}, "
            f"active={self.is_active}"
            f")>"
        )

    def to_dict(self) -> dict:
        """Convierte el grupo a diccionario para serialización."""
        return {
            "id": str(self.id),
            "group_name": self.group_name,
            "group_slug": self.group_slug,
            "description": self.description,
            "necesidad_l1": self.necesidad_l1,
            "subcategory_l2": self.subcategory_l2,
            "source": self.source,
            "is_active": self.is_active,
            "product_count": self.product_count,
            "brand_count": self.brand_count,
            "total_sales_amount": float(self.total_sales_amount) if self.total_sales_amount else 0,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
            "created_by": self.created_by,
        }

    def update_stats(self, session) -> None:
        """Actualiza estadísticas denormalizadas del grupo."""
        from sqlalchemy import func as sqla_func

        # Contar productos y marcas únicas
        stats = (
            session.query(
                sqla_func.count(InterchangeableGroupMember.id).label("products"),
                sqla_func.count(sqla_func.distinct(InterchangeableGroupMember.detected_brand)).label("brands"),
            )
            .filter(InterchangeableGroupMember.group_id == self.id)
            .first()
        )

        if stats:
            self.product_count = stats.products or 0
            self.brand_count = stats.brands or 0


class InterchangeableGroupMember(Base):
    """
    Miembros de un grupo intercambiable curado.

    Cada miembro representa un producto específico identificado por
    EAN13 y/o código nacional.
    """

    __tablename__ = "interchangeable_group_members"

    # === CLAVE PRIMARIA ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # === RELACIÓN CON GRUPO ===
    group_id = Column(
        UUID(as_uuid=True),
        ForeignKey("interchangeable_curated_groups.id", ondelete="CASCADE"),
        nullable=False,
        index=True
    )

    # === IDENTIFICACIÓN DEL PRODUCTO ===
    ean13 = Column(
        String(13),
        nullable=True,
        index=True,
        comment="Código EAN13 del producto"
    )
    codigo_nacional = Column(
        String(20),
        nullable=True,
        index=True,
        comment="Código nacional español"
    )
    product_name = Column(
        String(255),
        nullable=True,
        comment="Nombre del producto (cache para display)"
    )
    detected_brand = Column(
        String(100),
        nullable=True,
        comment="Marca detectada del producto"
    )

    # === METADATA ===
    added_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        nullable=False
    )
    added_by = Column(
        String(100),
        nullable=True,
        comment="Usuario que añadió el producto"
    )

    # === RELACIONES ===
    group = relationship(
        "InterchangeableCuratedGroup",
        back_populates="members"
    )

    # === CONFIGURACIÓN DE TABLA ===
    __table_args__ = (
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        return (
            f"<InterchangeableGroupMember("
            f"id={self.id}, "
            f"ean='{self.ean13}', "
            f"cn='{self.codigo_nacional}', "
            f"brand='{self.detected_brand}'"
            f")>"
        )

    def to_dict(self) -> dict:
        """Convierte el miembro a diccionario."""
        return {
            "id": str(self.id),
            "group_id": str(self.group_id),
            "ean13": self.ean13,
            "codigo_nacional": self.codigo_nacional,
            "product_name": self.product_name,
            "detected_brand": self.detected_brand,
            "added_at": self.added_at.isoformat() if self.added_at else None,
            "added_by": self.added_by,
        }
