# backend/app/models/intercambiable_group.py
"""
Modelo IntercambiableGroup - Grupos de productos sustituibles

Issue #446: Taxonomía jerárquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Este modelo representa el Nivel 4 de la jerarquía:
- Nivel 1: NECESIDAD (ml_category en SalesEnrichment)
- Nivel 2: Subcategoría (ml_subcategory en SalesEnrichment)
- Nivel 3: Línea de producto (brand_line en SalesEnrichment)
- Nivel 4: Grupo intercambiable (este modelo)

Un grupo intercambiable contiene productos de múltiples marcas
que pueden sustituirse entre sí (ej: fotoprotectores faciales SPF50+).
"""

import uuid
from datetime import datetime, timezone
from typing import TYPE_CHECKING

from sqlalchemy import Boolean, Column, DateTime, Integer, Numeric, String, Text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from .base import Base

if TYPE_CHECKING:
    from .sales_enrichment import SalesEnrichment


class IntercambiableGroup(Base):
    """
    Grupos de productos sustituibles (intercambiables) multi-marca.

    Requisitos para formar un grupo:
    - Mínimo 2 marcas diferentes (configurable via min_brands)
    - Mínimo 3 productos (configurable via min_products)
    - Validación manual por farmacéutico (validated=True)

    Ejemplos de grupos:
    - "Fotoprotectores faciales SPF50+" (ISDIN, Heliocare, Avene)
    - "Leches infantiles etapa 1" (Blemil, Pedialac, Almirón)
    - "Adhesivos para prótesis dental" (Corega, Kukident, Lacer)
    """

    __tablename__ = "intercambiable_group"

    # === CLAVE PRIMARIA ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # === IDENTIFICACIÓN ===
    name = Column(
        String(200),
        nullable=False,
        comment="Nombre descriptivo del grupo (ej: 'Fotoprotectores faciales SPF50+')"
    )
    slug = Column(
        String(100),
        unique=True,
        index=True,
        nullable=False,
        comment="Identificador URL-friendly único (ej: 'fotoprotectores-faciales-spf50')"
    )
    description = Column(
        Text,
        nullable=True,
        comment="Descripción detallada del grupo y criterios de intercambiabilidad"
    )

    # === CLASIFICACIÓN (Niveles 1-2 de jerarquía) ===
    necesidad = Column(
        String(100),
        nullable=False,
        index=True,
        comment="NECESIDAD principal (ej: 'proteccion_solar'). Nivel 1."
    )
    subcategory = Column(
        String(100),
        nullable=True,
        index=True,
        comment="Subcategoría (ej: 'facial', 'corporal'). Nivel 2."
    )

    # === CRITERIOS DE FORMACIÓN ===
    min_brands = Column(
        Integer,
        default=2,
        nullable=False,
        comment="Mínimo de marcas diferentes requeridas para el grupo"
    )
    min_products = Column(
        Integer,
        default=3,
        nullable=False,
        comment="Mínimo de productos requeridos para el grupo"
    )
    clustering_method = Column(
        String(50),
        nullable=True,
        comment="Método usado para crear el grupo: 'semantic', 'rule', 'manual'"
    )
    clustering_params = Column(
        Text,
        nullable=True,
        comment="Parámetros del clustering (JSON serializado)"
    )

    # === VALIDACIÓN ===
    validated = Column(
        Boolean,
        default=False,
        index=True,
        comment="True si un farmacéutico ha validado la intercambiabilidad"
    )
    validated_by = Column(
        String(100),
        nullable=True,
        comment="Email o nombre del farmacéutico que validó"
    )
    validated_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha/hora de validación"
    )
    validation_notes = Column(
        Text,
        nullable=True,
        comment="Notas del farmacéutico sobre la validación"
    )

    # === ESTADÍSTICAS (DENORMALIZADAS) ===
    product_count = Column(
        Integer,
        default=0,
        comment="Número de productos únicos en el grupo"
    )
    brand_count = Column(
        Integer,
        default=0,
        comment="Número de marcas diferentes en el grupo"
    )
    total_sales_amount = Column(
        Numeric(12, 2),
        default=0,
        comment="Suma total de ventas de productos del grupo"
    )
    total_sales_count = Column(
        Integer,
        default=0,
        comment="Número total de registros de venta en el grupo"
    )

    # === TIMESTAMPS ===
    created_at = Column(
        DateTime(timezone=True),
        server_default=func.now(),
        nullable=False,
        comment="Fecha de creación del grupo"
    )
    updated_at = Column(
        DateTime(timezone=True),
        onupdate=func.now(),
        nullable=True,
        comment="Última actualización"
    )

    # === RELACIONES ===
    sales_enrichments = relationship(
        "SalesEnrichment",
        back_populates="intercambiable_group",
        lazy="dynamic"
    )

    # === CONFIGURACIÓN DE TABLA ===
    __table_args__ = (
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        return (
            f"<IntercambiableGroup("
            f"id={self.id}, "
            f"name='{self.name}', "
            f"necesidad='{self.necesidad}', "
            f"validated={self.validated}, "
            f"products={self.product_count}, "
            f"brands={self.brand_count}"
            f")>"
        )

    def to_dict(self) -> dict:
        """Convierte el grupo a diccionario para serialización."""
        return {
            "id": str(self.id),
            "name": self.name,
            "slug": self.slug,
            "description": self.description,
            "necesidad": self.necesidad,
            "subcategory": self.subcategory,
            "min_brands": self.min_brands,
            "min_products": self.min_products,
            "clustering_method": self.clustering_method,
            "validated": self.validated,
            "validated_by": self.validated_by,
            "validated_at": self.validated_at.isoformat() if self.validated_at else None,
            "product_count": self.product_count,
            "brand_count": self.brand_count,
            "total_sales_amount": float(self.total_sales_amount) if self.total_sales_amount else 0,
            "total_sales_count": self.total_sales_count,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def validate(self, validated_by: str, notes: str = None) -> None:
        """
        Marca el grupo como validado por un farmacéutico.

        Args:
            validated_by: Email o nombre del farmacéutico
            notes: Notas opcionales sobre la validación
        """
        self.validated = True
        self.validated_by = validated_by
        self.validated_at = datetime.now(timezone.utc)
        self.validation_notes = notes

    def invalidate(self, reason: str = None) -> None:
        """
        Marca el grupo como no validado (rechazado).

        Args:
            reason: Razón del rechazo
        """
        self.validated = False
        self.validation_notes = f"Rechazado: {reason}" if reason else None

    def update_stats(self, session) -> None:
        """
        Actualiza estadísticas denormalizadas del grupo.

        Args:
            session: Sesión de SQLAlchemy
        """
        from sqlalchemy import func as sqla_func
        from .sales_data import SalesData

        # Contar productos únicos y marcas
        stats = (
            session.query(
                sqla_func.count(sqla_func.distinct(SalesData.product_name)).label("products"),
                sqla_func.count(sqla_func.distinct(SalesEnrichment.detected_brand)).label("brands"),
                sqla_func.sum(SalesData.total_amount).label("amount"),
                sqla_func.count(SalesData.id).label("count"),
            )
            .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
            .filter(SalesEnrichment.intercambiable_group_id == self.id)
            .first()
        )

        if stats:
            self.product_count = stats.products or 0
            self.brand_count = stats.brands or 0
            self.total_sales_amount = stats.amount or 0
            self.total_sales_count = stats.count or 0


# Import para type hints (evita circular imports)
from .sales_enrichment import SalesEnrichment  # noqa: E402, F401
