# backend/app/models/insight_feedback.py
"""
Modelo para almacenar feedback de usuarios sobre insights generados.

Issue #506: Insight Engine v2.0 - Motor de detección automática de anomalías.

Filosofía: El feedback permite filtrar insights que el usuario ya vio/actuó,
evitando ruido y permitiendo medir la efectividad del motor.

Acciones:
- SNOOZE: Posponer insight por N días (default 7)
- DISMISS: Descartar permanentemente (reconocido pero no accionado)
- RESOLVE: Marcar como resuelto (acción tomada)
"""

from datetime import datetime, timezone
from uuid import uuid4

import enum

from sqlalchemy import Column, DateTime, ForeignKey, Index, String, Text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base


class InsightCategory(str, enum.Enum):
    """Categoría del insight para agrupación en UI."""
    STOCK = "stock"           # STOCK_001, 002, 003
    MARGIN = "margin"         # MARGIN_001, 002
    HHI = "hhi"               # HHI_001 (dependencia de proveedor)
    TREND = "trend"           # TREND_001 (caídas sostenidas)
    SURTIDO = "surtido"       # SURTIDO_001, 002, 003


class InsightSeverity(str, enum.Enum):
    """Severidad del insight para priorización visual."""
    HIGH = "high"       # Rojo - Requiere acción inmediata
    MEDIUM = "medium"   # Naranja - Requiere atención
    LOW = "low"         # Azul - Informativo/oportunidad


class InsightFeedbackAction(str, enum.Enum):
    """Acción del usuario sobre un insight."""
    SNOOZE = "snooze"       # Posponer (default 7 días)
    DISMISS = "dismiss"     # Descartar permanentemente
    RESOLVE = "resolve"     # Marcar como resuelto


class InsightFeedback(Base):
    """
    Registro de feedback del usuario sobre insights generados.

    El insight_hash permite identificar el mismo insight a través del tiempo,
    incluso si los datos subyacentes cambian ligeramente.
    """
    __tablename__ = "insight_feedbacks"

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid4)

    # Foreign key to pharmacy
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True
    )

    # Insight identification
    insight_rule_code = Column(
        String(20),
        nullable=False,
        comment="Código de la regla (ej: STOCK_001, MARGIN_002)"
    )
    insight_hash = Column(
        String(64),
        nullable=False,
        comment="SHA256[:16] de affected_items para deduplicación"
    )

    # User action (String to avoid SQLAlchemy enum issues in multi-worker)
    action = Column(
        String(20),
        nullable=False,
        comment="snooze, dismiss, resolve"
    )

    # Snooze details
    snoozed_until = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha hasta la que el insight está pospuesto"
    )

    # User notes
    notes = Column(
        Text,
        nullable=True,
        comment="Notas opcionales del usuario"
    )

    # Timestamps
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False
    )

    # Relationships
    pharmacy = relationship("Pharmacy", lazy="selectin")

    # Indexes for common queries
    __table_args__ = (
        # Composite index for looking up active feedbacks by pharmacy
        Index(
            "ix_insight_feedback_pharmacy_rule",
            "pharmacy_id",
            "insight_rule_code"
        ),
        # Index for finding specific insight by hash
        Index(
            "ix_insight_feedback_hash",
            "insight_hash"
        ),
        # Index for cleanup of expired snoozes
        Index(
            "ix_insight_feedback_snoozed_until",
            "snoozed_until",
            postgresql_where=(snoozed_until.isnot(None))
        ),
    )

    def __repr__(self):
        return (
            f"<InsightFeedback(id={self.id}, "
            f"rule={self.insight_rule_code}, "
            f"action={self.action})>"
        )

    @property
    def is_snooze(self) -> bool:
        """True if this feedback is a snooze action."""
        return self.action == InsightFeedbackAction.SNOOZE.value

    @property
    def is_dismiss(self) -> bool:
        """True if this feedback is a dismiss action."""
        return self.action == InsightFeedbackAction.DISMISS.value

    @property
    def is_resolve(self) -> bool:
        """True if this feedback is a resolve action."""
        return self.action == InsightFeedbackAction.RESOLVE.value

    @property
    def is_snooze_expired(self) -> bool:
        """True if snooze has expired (insight should reappear)."""
        if not self.is_snooze or self.snoozed_until is None:
            return False
        return datetime.now(timezone.utc) > self.snoozed_until

    @property
    def is_active_suppression(self) -> bool:
        """True if this feedback should suppress the insight from showing."""
        if self.is_dismiss or self.is_resolve:
            return True
        if self.is_snooze and not self.is_snooze_expired:
            return True
        return False
