# backend/app/models/homogeneous_group_master.py
"""
Modelo HomogeneousGroupMaster - Catálogo maestro de conjuntos homogéneos
Contiene datos oficiales del nomenclator sin vinculación a farmacias específicas
"""

import uuid

from sqlalchemy import Boolean, Column, DateTime, Index, Integer, Numeric, String
from sqlalchemy.dialects.postgresql import JSONB, UUID

from app.utils.datetime_utils import utc_now

from .base import Base


class HomogeneousGroupMaster(Base):
    """
    Catálogo maestro de conjuntos homogéneos basado en datos oficiales del nomenclator.

    Esta tabla contiene:
    - Datos oficiales de conjuntos homogéneos
    - PVL de referencia calculado del nomenclator
    - Información terapéutica y regulatoria
    - Datos de laboratorios disponibles en cada conjunto

    NO contiene:
    - Datos específicos de farmacias (pharmacy_id)
    - Métricas de ventas
    - Información de partners (eso va en PharmacyHomogeneousMetrics)
    """

    __tablename__ = "homogeneous_groups_master"
    __table_args__ = (
        # Índices para consultas del catálogo
        Index("ix_master_homogeneous_code", "homogeneous_code"),
        Index("ix_master_active_ingredient", "main_active_ingredient"),
        Index("ix_master_updated", "updated_at"),
        {"extend_existing": True},
    )

    # === CAMPOS IDENTIFICADORES ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)
    homogeneous_code = Column(String(50), nullable=False, unique=True, index=True)  # nomen_codigo_homogeneo
    homogeneous_name = Column(String(200), nullable=True)  # nomen_nombre_homogeneo

    # === DATOS DE PRECIOS DE REFERENCIA ===
    # Estos datos vienen del nomenclator oficial
    reference_pvp = Column(Numeric(10, 4), nullable=True)  # PVP de referencia oficial
    calculated_pvl = Column(Numeric(10, 4), nullable=True)  # PVL calculado del nomenclator
    min_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP mínimo del conjunto
    max_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP máximo del conjunto
    avg_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP promedio del conjunto

    # === INFORMACIÓN DE LABORATORIOS DISPONIBLES ===
    # Datos de laboratorios que comercializan productos en este conjunto
    available_laboratories = Column(JSONB, nullable=True)  # Lista de todos los labs: ["NORMON S.A.", "CINFA", ...]
    total_labs_in_group = Column(Integer, default=0)  # Total laboratorios en el conjunto
    total_products_in_group = Column(Integer, default=0)  # Total productos en el conjunto

    # === INFORMACIÓN TERAPÉUTICA ===
    main_active_ingredient = Column(String(500), nullable=True, index=True)  # Principio activo principal
    active_ingredients_list = Column(JSONB, nullable=True)  # Lista completa de principios activos
    therapeutic_group = Column(String(100), nullable=True)  # Grupo terapéutico
    atc_code = Column(String(20), nullable=True)  # Código ATC principal
    requires_prescription = Column(Boolean, nullable=True)  # Requiere prescripción

    # === DATOS REGULATORIOS ===
    is_efg = Column(Boolean, nullable=True)  # Es genérico (EFG)
    pharmaceutical_form = Column(String(100), nullable=True)  # Forma farmacéutica
    route_of_administration = Column(String(100), nullable=True)  # Vía de administración

    # === METADATOS DEL CATÁLOGO ===
    nomenclator_data = Column(JSONB, nullable=True)  # Datos adicionales del nomenclator
    """
    Ejemplo nomenclator_data:
    {
        "nomenclator_update_date": "2024-01-15",
        "data_sources": ["nomenclator", "cima"],
        "validation_status": "validated",
        "notes": "Conjunto completo",
        "regulatory_warnings": []
    }
    """

    # Control de sincronización con fuentes externas
    last_nomenclator_sync = Column(DateTime, nullable=True)  # Última sincronización con nomenclator
    last_cima_sync = Column(DateTime, nullable=True)  # Última sincronización con CIMA
    sync_status = Column(String(50), default="PENDING")  # SYNCHRONIZED, PENDING, ERROR

    # Timestamps de control
    created_at = Column(DateTime, default=utc_now, nullable=False)
    updated_at = Column(
        DateTime,
        default=utc_now,
        onupdate=utc_now,
        nullable=False,
        index=True,
    )

    def __repr__(self):
        return f"<HomogeneousGroupMaster {self.homogeneous_code}: {self.homogeneous_name}>"

    def to_dict(self, include_metadata: bool = False) -> dict:
        """
        Convierte a diccionario para APIs del catálogo maestro

        Args:
            include_metadata: Si incluir metadatos detallados
        """
        result = {
            "id": str(self.id),
            "homogeneous_code": self.homogeneous_code,
            "homogeneous_name": self.homogeneous_name,
            "calculated_pvl": (float(self.calculated_pvl) if self.calculated_pvl else 0.0),
            "reference_pvp": float(self.reference_pvp) if self.reference_pvp else 0.0,
            # Datos de laboratorios
            "available_laboratories": self.available_laboratories or [],
            "total_labs_in_group": self.total_labs_in_group,
            "total_products_in_group": self.total_products_in_group,
            # Información terapéutica
            "main_active_ingredient": self.main_active_ingredient,
            "therapeutic_group": self.therapeutic_group,
            "atc_code": self.atc_code,
            "requires_prescription": self.requires_prescription,
            "is_efg": self.is_efg,
            # Control de sincronización
            "sync_status": self.sync_status,
            "updated_at": self.updated_at.isoformat(),
        }

        if include_metadata:
            result.update(
                {
                    "nomenclator_data": self.nomenclator_data,
                    "last_nomenclator_sync": (
                        self.last_nomenclator_sync.isoformat() if self.last_nomenclator_sync else None
                    ),
                    "last_cima_sync": (self.last_cima_sync.isoformat() if self.last_cima_sync else None),
                    "min_pvp_in_group": (float(self.min_pvp_in_group) if self.min_pvp_in_group else 0.0),
                    "max_pvp_in_group": (float(self.max_pvp_in_group) if self.max_pvp_in_group else 0.0),
                    "avg_pvp_in_group": (float(self.avg_pvp_in_group) if self.avg_pvp_in_group else 0.0),
                    "active_ingredients_list": self.active_ingredients_list,
                    "pharmaceutical_form": self.pharmaceutical_form,
                    "route_of_administration": self.route_of_administration,
                }
            )

        return result

    @property
    def has_multiple_laboratories(self) -> bool:
        """
        Verifica si el conjunto homogéneo tiene múltiples laboratorios
        (condición necesaria para análisis de genéricos)
        """
        return self.total_labs_in_group > 1

    def get_laboratory_list(self) -> list:
        """
        Obtiene lista limpia de laboratorios disponibles
        """
        if not self.available_laboratories:
            return []

        # Asegurar que es una lista y filtrar valores nulos/vacíos
        labs = self.available_laboratories if isinstance(self.available_laboratories, list) else []
        return [lab for lab in labs if lab and lab.strip()]
