# backend/app/models/homogeneous_group.py
"""
Modelo HomogeneousGroup - Tabla auxiliar optimizada para análisis de genéricos
Almacena datos precalculados de conjuntos homogéneos con PVL y estadísticas partners
"""

import uuid

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, Index, Integer, Numeric, String
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.utils.datetime_utils import utc_now

from .base import Base


class HomogeneousGroup(Base):
    """
    Conjuntos homogéneos de medicamentos con datos precalculados para análisis eficiente.

    Esta tabla auxiliar optimiza consultas de genéricos almacenando:
    - PVL calculado automáticamente del conjunto
    - Estadísticas de laboratorios partners
    - Métricas de ventas agregadas por farmacia

    Se actualiza automáticamente cuando:
    - Se cargan nuevos datos del nomenclator
    - Se modifican laboratorios partners
    - Se procesan nuevos datos de ventas
    """

    __tablename__ = "homogeneous_groups"
    __table_args__ = (
        # Índices compuestos para consultas frecuentes
        Index("ix_homogeneous_pharmacy_code", "pharmacy_id", "homogeneous_code"),
        Index("ix_homogeneous_partners", "pharmacy_id", "has_partner_labs"),
        Index("ix_homogeneous_updated", "updated_at"),
        {"extend_existing": True},
    )

    # === CAMPOS IDENTIFICADORES ===
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )
    homogeneous_code = Column(String(50), nullable=False, index=True)  # nomen_codigo_homogeneo
    homogeneous_name = Column(String(200), nullable=True)  # nomen_nombre_homogeneo

    # === DATOS DE PRECIOS PRECALCULADOS ===
    representative_pvp = Column(Numeric(10, 4), nullable=True)  # PVP representativo del conjunto
    calculated_pvl = Column(Numeric(10, 4), nullable=True)  # PVL calculado automáticamente
    min_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP mínimo del conjunto
    max_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP máximo del conjunto
    avg_pvp_in_group = Column(Numeric(10, 4), nullable=True)  # PVP promedio del conjunto

    # === INFORMACIÓN DE LABORATORIOS PARTNERS ===
    has_partner_labs = Column(Boolean, default=False, index=True)  # Tiene laboratorios partners
    partner_laboratories = Column(JSONB, nullable=True)  # Lista de labs partners: ["NORMON S.A.", "CINFA", ...]
    total_labs_in_group = Column(Integer, default=0)  # Total laboratorios en el conjunto
    partner_labs_count = Column(Integer, default=0)  # Cantidad de partners en el conjunto

    # === ESTADÍSTICAS DE VENTAS AGREGADAS ===
    # Total ventas del conjunto homogéneo (última actualización)
    total_units_sold = Column(Integer, default=0)  # Total unidades vendidas
    total_transactions = Column(Integer, default=0)  # Total transacciones
    total_revenue = Column(Numeric(12, 2), default=0)  # Ingresos totales €

    # Ventas de laboratorios partners
    partner_units_sold = Column(Integer, default=0)  # Unidades de partners
    partner_revenue = Column(Numeric(12, 2), default=0)  # Ingresos de partners €

    # Ventas de laboratorios NO partners (oportunidad de descuento)
    non_partner_units_sold = Column(Integer, default=0, index=True)  # CLAVE: Unidades NO-partner
    non_partner_revenue = Column(Numeric(12, 2), default=0)  # Ingresos NO-partners €
    non_partner_pvl_base = Column(Numeric(12, 2), default=0)  # PVL base para cálculo descuento

    # === INFORMACIÓN TERAPÉUTICA ===
    main_active_ingredient = Column(String(500), nullable=True)  # Principio activo principal
    therapeutic_group = Column(String(100), nullable=True)  # Grupo terapéutico
    requires_prescription = Column(Boolean, nullable=True)  # Requiere prescripción

    # === METADATOS Y CONTROL ===
    data_period_start = Column(DateTime, nullable=True)  # Período de datos (inicio)
    data_period_end = Column(DateTime, nullable=True)  # Período de datos (fin)
    last_sales_date = Column(DateTime, nullable=True)  # Última venta registrada

    calculation_metadata = Column(JSONB, nullable=True)  # Metadatos del cálculo
    """
    Ejemplo calculation_metadata:
    {
        "pvl_calculation_method": "nomenclator_standard",
        "partner_discount_applied": false,
        "data_completeness_score": 0.95,
        "products_in_group": 12,
        "enrichment_sources": ["nomenclator", "cima"],
        "last_nomenclator_update": "2024-01-15",
        "calculation_warnings": []
    }
    """

    # Timestamps de control
    created_at = Column(DateTime, default=utc_now, nullable=False)
    updated_at = Column(
        DateTime,
        default=utc_now,
        onupdate=utc_now,
        nullable=False,
        index=True,
    )

    # === RELACIONES ===
    pharmacy = relationship("Pharmacy", back_populates="homogeneous_groups")

    def __repr__(self):
        return f"<HomogeneousGroup {self.homogeneous_code}: {self.non_partner_units_sold} units, €{self.calculated_pvl} PVL>"

    def to_dict(self, include_metadata: bool = False) -> dict:
        """
        Convierte a diccionario para APIs

        Args:
            include_metadata: Si incluir metadatos de cálculo
        """
        result = {
            "id": str(self.id),
            "homogeneous_code": self.homogeneous_code,
            "homogeneous_name": self.homogeneous_name,
            "calculated_pvl": (float(self.calculated_pvl) if self.calculated_pvl else 0.0),
            "representative_pvp": (float(self.representative_pvp) if self.representative_pvp else 0.0),
            # Datos partners
            "has_partner_labs": self.has_partner_labs,
            "partner_laboratories": self.partner_laboratories or [],
            "partner_labs_count": self.partner_labs_count,
            # Datos de oportunidad (clave para descuentos)
            "non_partner_units_sold": self.non_partner_units_sold,
            "non_partner_pvl_base": (float(self.non_partner_pvl_base) if self.non_partner_pvl_base else 0.0),
            # Estadísticas agregadas
            "total_units_sold": self.total_units_sold,
            "total_revenue": float(self.total_revenue) if self.total_revenue else 0.0,
            # Información terapéutica
            "main_active_ingredient": self.main_active_ingredient,
            "requires_prescription": self.requires_prescription,
            # Control temporal
            "last_sales_date": (self.last_sales_date.isoformat() if self.last_sales_date else None),
            "updated_at": self.updated_at.isoformat(),
        }

        if include_metadata:
            result["calculation_metadata"] = self.calculation_metadata
            result["data_period_start"] = self.data_period_start.isoformat() if self.data_period_start else None
            result["data_period_end"] = self.data_period_end.isoformat() if self.data_period_end else None

        return result

    @property
    def potential_savings_base(self) -> float:
        """
        Base de ahorro potencial (sin aplicar % descuento).
        El frontend multiplica por el % del slider.

        Returns:
            float: PVL base de unidades no-partner para cálculo de descuento
        """
        return float(self.non_partner_pvl_base or 0)

    @property
    def partner_penetration(self) -> float:
        """
        Penetración de laboratorios partners en el conjunto (0-100%)

        Returns:
            float: Porcentaje de ventas que son de partners
        """
        if self.total_units_sold == 0:
            return 0.0
        return round((self.partner_units_sold / self.total_units_sold) * 100, 2)

    def calculate_savings_with_discount(self, discount_percentage: float) -> dict:
        """
        Calcula ahorro potencial aplicando descuento sobre la base PVL

        Args:
            discount_percentage: Descuento a aplicar (ej: 20.0 para 20%)

        Returns:
            dict: Información de ahorro calculado
        """
        if not self.non_partner_pvl_base or discount_percentage <= 0:
            return {
                "savings_amount": 0.0,
                "units_affected": self.non_partner_units_sold,
                "discount_applied": discount_percentage,
                "base_pvl": float(self.non_partner_pvl_base or 0),
            }

        base_pvl = float(self.non_partner_pvl_base)
        savings = base_pvl * (discount_percentage / 100)

        return {
            "savings_amount": round(savings, 2),
            "units_affected": self.non_partner_units_sold,
            "discount_applied": discount_percentage,
            "base_pvl": base_pvl,
            "savings_per_unit": round(savings / max(self.non_partner_units_sold, 1), 2),
        }
