# backend/app/models/file_upload.py
import enum
import uuid

from sqlalchemy import Column, DateTime, Enum, ForeignKey, Integer, String, Text
from sqlalchemy.dialects.postgresql import JSON, UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class UploadStatus(str, enum.Enum):
    """Estados posibles de un upload"""

    PENDING = "pending"
    QUEUED = "queued"  # NEW (Issue #332): Archivo en cola para procesamiento
    PARSING = "parsing"  # NEW (Issue #329): Parseando archivo ERP
    VALIDATING = "validating"  # NEW (Issue #329): Validando datos
    SAVING = "saving"  # NEW (Issue #329): Guardando en base de datos
    PROCESSING = "processing"  # Enriqueciendo con CIMA/nomenclator
    COMPLETED = "completed"
    ERROR = "error"
    PARTIAL = "partial"


class FileType(str, enum.Enum):
    """Tipos de archivo/ERP soportados"""

    # Ventas
    FARMATIC = "farmatic"
    FARMANAGER = "farmanager"
    NIXFARMA = "nixfarma"
    UNYCOP = "unycop"
    IOFWIN = "iofwin"
    # Inventario (Issue #476)
    INVENTORY_FARMATIC = "inventory_farmatic"
    INVENTORY_FARMANAGER = "inventory_farmanager"
    # Default
    UNKNOWN = "unknown"


class FileUpload(Base):
    """Modelo para registrar uploads de archivos"""

    __tablename__ = "file_uploads"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Foreign keys
    pharmacy_id = Column(UUID(as_uuid=True), ForeignKey("pharmacies.id", ondelete="CASCADE"))
    user_id = Column(UUID(as_uuid=True), ForeignKey("users.id", ondelete="SET NULL"), nullable=True)

    # Información del archivo
    filename = Column(String(255), nullable=False)
    original_filename = Column(String(255))
    file_path = Column(String(500))  # Ruta en el servidor
    file_type = Column(Enum(FileType), default=FileType.UNKNOWN)
    file_size = Column(Integer)  # En bytes

    # Estado del procesamiento
    status = Column(Enum(UploadStatus), default=UploadStatus.PENDING)
    error_message = Column(Text)  # Errores fatales que bloquean el proceso
    warnings = Column(JSON, default=list)  # Issue #412: Avisos no-bloqueantes (encoding fallback, etc.)
    processing_notes = Column(Text)

    # Estadísticas del procesamiento
    rows_total = Column(Integer, default=0)
    rows_processed = Column(Integer, default=0)
    rows_with_errors = Column(Integer, default=0)
    rows_duplicates = Column(Integer, default=0)  # Issue #330: Duplicados detectados por ventana temporal

    # Timestamps
    uploaded_at = Column(DateTime, default=utc_now)
    processing_started_at = Column(DateTime)
    processing_completed_at = Column(DateTime)

    # Relaciones con otras tablas
    pharmacy = relationship("Pharmacy", back_populates="uploads")
    user = relationship("User", back_populates="uploads")
    sales_data = relationship("SalesData", back_populates="upload", cascade="all, delete-orphan")
    inventory_snapshots = relationship(
        "InventorySnapshot", back_populates="file_upload", cascade="all, delete-orphan"
    )  # Issue #476

    def __repr__(self):
        return f"<FileUpload {self.filename} - {self.status.value}>"

    def to_dict(self):
        """Convierte el modelo a diccionario para APIs"""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id) if self.pharmacy_id else None,
            "user_id": str(self.user_id) if self.user_id else None,
            "filename": self.filename,
            "original_filename": self.original_filename,
            "file_type": self.file_type.value if self.file_type else "unknown",
            "file_size": self.file_size,
            "status": self.status.value if self.status else "unknown",
            "error_message": self.error_message,
            "warnings": self.warnings or [],  # Issue #412
            "processing_notes": self.processing_notes,
            "rows_total": self.rows_total,
            "rows_processed": self.rows_processed,
            "rows_with_errors": self.rows_with_errors,
            "rows_duplicates": self.rows_duplicates,  # Issue #330
            "uploaded_at": self.uploaded_at.isoformat() if self.uploaded_at else None,
            "processing_started_at": (self.processing_started_at.isoformat() if self.processing_started_at else None),
            "processing_completed_at": (
                self.processing_completed_at.isoformat() if self.processing_completed_at else None
            ),
        }
