# backend/app/models/erp_sync_state.py
"""
ERP Sync State Model (Simplified for Local Installation).

Pivot 2026: Local single-tenant architecture means:
- ONE pharmacy per installation
- ONE ERP per installation
- Config in .env (connection details)
- Only SYNC STATE in database (for incremental sync)

This model tracks ONLY the sync state, not the configuration.
Configuration comes from environment variables.
"""
import enum
from datetime import datetime
from typing import Optional

from sqlalchemy import Boolean, Column, DateTime, Integer, String, Text

from app.database import Base
from app.utils.datetime_utils import utc_now


class SyncStatusEnum(str, enum.Enum):
    """Status of sync operations."""

    IDLE = "idle"  # Ready for next sync
    SYNCING = "syncing"  # Currently running
    ERROR = "error"  # Last sync failed
    DISABLED = "disabled"  # Manually disabled


class ERPSyncState(Base):
    """
    ERP Sync State for local installation.

    Single-row table (id=1 always) tracking sync progress.
    Configuration comes from environment variables:

    Environment Variables:
        ERP_TYPE: farmanager | farmatic | nixfarma | unycop
        ERP_HOST: Database host (default: localhost)
        ERP_PORT: Database port (default: 3306 for MySQL, 1433 for SQL Server)
        ERP_DATABASE: Database name (default: gesql for Farmanager)
        ERP_USER: Database user
        ERP_PASSWORD: Database password
        ERP_SYNC_INTERVAL: Sync interval in minutes (default: 15)

    Usage:
        # Get or create the singleton state
        state = db.query(ERPSyncState).first()
        if not state:
            state = ERPSyncState(id=1)
            db.add(state)

        # Check if should sync
        if state.should_sync(interval_minutes=15):
            state.mark_sync_started()
            # ... do sync ...
            state.mark_sync_completed(records=100, last_timestamp=max_ts)
    """

    __tablename__ = "erp_sync_state"
    __table_args__ = {"extend_existing": True}

    # Single-row table: id is always 1
    id = Column(Integer, primary_key=True, default=1)

    # Current status
    status = Column(String(20), default=SyncStatusEnum.IDLE.value)
    sync_enabled = Column(Boolean, default=True)

    # Sync timestamps
    last_sync_started_at = Column(DateTime(timezone=True))
    last_sync_completed_at = Column(DateTime(timezone=True))

    # High water mark for incremental sync (CRITICAL)
    # This is the max sale timestamp we've seen
    last_sale_timestamp = Column(DateTime(timezone=True))

    # Last successful sync stats
    last_sync_records = Column(Integer, default=0)
    last_sync_duration_seconds = Column(Integer, default=0)

    # Cumulative stats
    total_syncs = Column(Integer, default=0)
    total_records_synced = Column(Integer, default=0)

    # Error tracking
    consecutive_errors = Column(Integer, default=0)
    last_error = Column(Text)
    last_error_at = Column(DateTime(timezone=True))

    # Metadata
    created_at = Column(DateTime(timezone=True), default=utc_now)
    updated_at = Column(DateTime(timezone=True), default=utc_now, onupdate=utc_now)

    def __repr__(self):
        return f"<ERPSyncState status={self.status} last_sync={self.last_sync_completed_at}>"

    def to_dict(self) -> dict:
        """Convert to dictionary for API responses."""
        return {
            "status": self.status,
            "sync_enabled": self.sync_enabled,
            "last_sync_completed_at": (
                self.last_sync_completed_at.isoformat()
                if self.last_sync_completed_at
                else None
            ),
            "last_sale_timestamp": (
                self.last_sale_timestamp.isoformat()
                if self.last_sale_timestamp
                else None
            ),
            "last_sync_records": self.last_sync_records,
            "last_sync_duration_seconds": self.last_sync_duration_seconds,
            "total_syncs": self.total_syncs,
            "total_records_synced": self.total_records_synced,
            "consecutive_errors": self.consecutive_errors,
            "last_error": self.last_error,
        }

    def should_sync(self, interval_minutes: int = 15) -> bool:
        """
        Check if sync should run based on interval and status.

        Args:
            interval_minutes: Minimum minutes between syncs

        Returns:
            True if enough time has passed and sync is enabled
        """
        if not self.sync_enabled:
            return False

        if self.status == SyncStatusEnum.DISABLED.value:
            return False

        if self.status == SyncStatusEnum.SYNCING.value:
            # Already syncing
            return False

        # Too many consecutive errors → require manual intervention
        if self.consecutive_errors >= 5:
            return False

        # Never synced → should sync
        if not self.last_sync_completed_at:
            return True

        # Check if enough time has passed
        minutes_since_sync = (
            utc_now() - self.last_sync_completed_at
        ).total_seconds() / 60

        return minutes_since_sync >= interval_minutes

    def mark_sync_started(self) -> None:
        """Mark sync as started."""
        self.last_sync_started_at = utc_now()
        self.status = SyncStatusEnum.SYNCING.value

    def mark_sync_completed(
        self,
        records_synced: int,
        duration_seconds: int,
        last_sale_timestamp: Optional[datetime] = None,
    ) -> None:
        """
        Mark sync as successfully completed.

        Args:
            records_synced: Number of records synced in this run
            duration_seconds: Duration of sync operation
            last_sale_timestamp: Max sale timestamp seen (for incremental)
        """
        now = utc_now()
        self.last_sync_completed_at = now
        self.last_sync_records = records_synced
        self.last_sync_duration_seconds = duration_seconds
        self.total_syncs += 1
        self.total_records_synced += records_synced
        self.consecutive_errors = 0  # Reset on success
        self.last_error = None
        self.status = SyncStatusEnum.IDLE.value

        if last_sale_timestamp:
            self.last_sale_timestamp = last_sale_timestamp

    def mark_sync_error(self, error_message: str) -> None:
        """Mark sync as failed."""
        now = utc_now()
        self.last_sync_completed_at = now
        self.last_error = error_message
        self.last_error_at = now
        self.consecutive_errors += 1
        self.status = SyncStatusEnum.ERROR.value

    def reset_errors(self) -> None:
        """Reset error counters (for manual intervention)."""
        self.consecutive_errors = 0
        self.last_error = None
        self.status = SyncStatusEnum.IDLE.value
