# backend/app/models/dim_date.py
"""
Modelo para dimensión de fechas (Date Dimension).

Issue #503: Centralizar lógica temporal para Time Intelligence.

Beneficios:
- Elimina EXTRACT() en runtime (pre-calculado)
- Nombres españoles centralizados (lunes, enero, etc.)
- Temporadas farmacéuticas pre-calculadas
- Festivos nacionales incluidos
- ISO-8601 para semanas (España usa lunes como inicio)

Uso:
    # JOIN en lugar de EXTRACT()
    SELECT d.weekday_name, SUM(s.quantity)
    FROM sales_data s
    JOIN dim_date d ON DATE(s.sale_date) = d.date_key
    GROUP BY d.weekday_name
"""

from sqlalchemy import Boolean, Column, Date, Index, Integer, String

from app.database import Base


class DimDate(Base):
    """
    Tabla de dimensión de fechas para analítica temporal.

    Cobertura: 2020-01-01 a 2030-12-31 (~4000 registros).
    Primary Key: date_key (DATE) - Natural key para JOINs eficientes.

    Atributos:
        Jerarquía temporal: year, quarter, month, week_of_year, day_of_month, weekday
        Nombres españoles: month_name, weekday_name (con versiones cortas)
        Flags: is_weekend, is_holiday
        Temporadas: is_flu_season, is_allergy_season, is_summer_season
        Claves de agrupación: year_month, year_quarter
    """

    __tablename__ = "dim_date"
    __table_args__ = (
        # Índice compuesto para reportes mensuales (feedback usuario)
        Index("idx_dim_date_year_month_composite", "year", "month"),
        {"extend_existing": True},
    )

    # Primary key = fecha (DATE, no UUID)
    # Natural key: más eficiente para JOINs con sale_date
    date_key = Column(Date, primary_key=True)

    # ===== Jerarquía temporal =====
    year = Column(Integer, nullable=False, index=True)
    quarter = Column(Integer, nullable=False)  # 1-4
    month = Column(Integer, nullable=False, index=True)  # 1-12
    week_of_year = Column(Integer, nullable=False)  # ISO-8601: 1-53
    day_of_month = Column(Integer, nullable=False)  # 1-31
    day_of_year = Column(Integer, nullable=False)  # 1-366
    weekday = Column(Integer, nullable=False, index=True)  # ISO-8601: 1=Lun, 7=Dom

    # ===== Nombres en español =====
    month_name = Column(String(20), nullable=False)  # "Enero", "Febrero"...
    month_name_short = Column(String(3), nullable=False)  # "Ene", "Feb"...
    weekday_name = Column(String(20), nullable=False)  # "Lunes", "Martes"...
    weekday_name_short = Column(String(3), nullable=False)  # "Lun", "Mar"...

    # ===== Flags de calendario =====
    is_weekend = Column(Boolean, nullable=False, default=False)
    is_holiday = Column(Boolean, nullable=False, default=False)
    holiday_name = Column(String(100), nullable=True)  # "Navidad", "Año Nuevo"

    # ===== Temporadas farmacéuticas =====
    # Basado en literatura médica española y patrones de venta
    is_flu_season = Column(Boolean, nullable=False, default=False)  # Oct-Mar
    is_allergy_season = Column(Boolean, nullable=False, default=False)  # Mar-Jun
    is_summer_season = Column(Boolean, nullable=False, default=False)  # Jul-Ago

    # ===== Claves de agrupación =====
    year_month = Column(String(7), nullable=False)  # "2025-01"
    year_quarter = Column(String(7), nullable=False)  # "2025-Q1"

    def __repr__(self) -> str:
        return f"<DimDate {self.date_key} ({self.weekday_name})>"

    def to_dict(self) -> dict:
        """Convierte el modelo a diccionario para APIs."""
        return {
            "date_key": self.date_key.isoformat() if self.date_key else None,
            "year": self.year,
            "quarter": self.quarter,
            "month": self.month,
            "week_of_year": self.week_of_year,
            "day_of_month": self.day_of_month,
            "day_of_year": self.day_of_year,
            "weekday": self.weekday,
            "month_name": self.month_name,
            "month_name_short": self.month_name_short,
            "weekday_name": self.weekday_name,
            "weekday_name_short": self.weekday_name_short,
            "is_weekend": self.is_weekend,
            "is_holiday": self.is_holiday,
            "holiday_name": self.holiday_name,
            "is_flu_season": self.is_flu_season,
            "is_allergy_season": self.is_allergy_season,
            "is_summer_season": self.is_summer_season,
            "year_month": self.year_month,
            "year_quarter": self.year_quarter,
        }

    @property
    def is_working_day(self) -> bool:
        """Retorna True si es día laborable (no fin de semana ni festivo)."""
        return not self.is_weekend and not self.is_holiday
