"""
CIMA Stall Event Model - Persistencia de eventos de stall en sincronización CIMA

Issue: #114 (Resiliencia CIMA)
Propósito: Registrar y analizar patrones de stall en sincronización CIMA
"""

import uuid
from datetime import datetime, timezone

from sqlalchemy import Column, DateTime, Integer
from sqlalchemy.dialects.postgresql import UUID

from app.database import Base


def utc_now():
    """Helper para timestamps UTC consistentes."""
    return datetime.now(timezone.utc)


class CIMAStallEvent(Base):
    """
    Evento de stall detectado en sincronización CIMA.

    Registra cada vez que el proceso de sincronización se estanca en una página
    sin progreso durante el threshold configurado (default: 120s).

    Casos de uso:
    - Análisis post-mortem de fallos de sincronización
    - Detección de patrones de páginas problemáticas
    - Métricas de confiabilidad del proceso CIMA
    - Debugging de configuración de thresholds

    Relacionado con:
    - SystemStatus (component='CIMA'): Estado actual de sincronización
    - ProductCatalog: Productos que se intentaban sincronizar
    """

    __tablename__ = "cima_stall_events"
    __table_args__ = {"extend_existing": True}  # Thread-safe para Render multi-worker

    id = Column(
        UUID(as_uuid=True), primary_key=True, default=uuid.uuid4, comment="Identificador único del evento de stall"
    )

    page_number = Column(Integer, nullable=False, index=True, comment="Número de página donde ocurrió el stall")

    stall_duration = Column(Integer, nullable=False, comment="Duración del stall en segundos antes de detección")

    chunks_attempted = Column(Integer, nullable=False, comment="Número de chunks consecutivos sin progreso")

    created_at = Column(
        DateTime(timezone=True), default=utc_now, nullable=False, index=True, comment="Timestamp del evento (UTC)"
    )

    def __repr__(self):
        return (
            f"<CIMAStallEvent(page={self.page_number}, "
            f"duration={self.stall_duration}s, "
            f"chunks={self.chunks_attempted})>"
        )

    def to_dict(self):
        """Serialización para APIs y logging."""
        return {
            "id": str(self.id),
            "page_number": self.page_number,
            "stall_duration": self.stall_duration,
            "chunks_attempted": self.chunks_attempted,
            "created_at": self.created_at.isoformat() if self.created_at else None,
        }
