# backend/app/models/category_alias.py
"""
Modelo para almacenar aliases de categorías del clasificador NECESIDAD.

Issue #459: Mover category_normalization.py a base de datos para mejor
mantenibilidad y trazabilidad.

Propósito:
    Mapear outputs del clasificador (ej: "aftas_llagas") a nombres en DB (ej: "aftas").
    Diferente de KeywordOverride: esto normaliza OUTPUT, no hace pattern matching de INPUT.

Beneficios vs archivo estático:
    - Cambios sin deploys
    - Audit trail (quién, cuándo, por qué)
    - Analytics de uso (usage_count, last_used_at)
"""

from datetime import datetime, timezone

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    Index,
    Integer,
    String,
)

from app.database import Base


class CategoryAlias(Base):
    """
    Alias de categoría para normalización de outputs del clasificador.

    Ejemplo:
        source_category="aftas_llagas" → target_category="aftas"
        (clasificador devuelve "aftas_llagas", pero DB espera "aftas")

    A diferencia de KeywordOverride:
        - Más simple (sin priority, sin keyword_type)
        - 1:1 mapping (un source → un target)
        - Incluye analytics de uso
    """
    __tablename__ = "category_aliases"

    id = Column(Integer, primary_key=True, index=True)

    # === ALIAS DATA ===
    # Lo que devuelve el clasificador
    source_category = Column(
        String(100),
        nullable=False,
        unique=True,
        comment="Categoría del clasificador (input a normalizar)"
    )

    # Lo que espera la base de datos
    target_category = Column(
        String(100),
        nullable=False,
        comment="Categoría normalizada (output)"
    )

    # === STATUS ===
    is_active = Column(
        Boolean,
        nullable=False,
        default=True,
        comment="Solo activos se usan en normalización"
    )

    # === METADATA ===
    reason = Column(
        String(500),
        nullable=True,
        comment="Razón del alias (ej: 'DB tiene 598 cicatrizacion vs 33 heridas_cicatrizacion')"
    )

    # === ANALYTICS ===
    usage_count = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Veces que se ha usado este alias"
    )

    last_used_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Última vez que se usó (detectar obsoletos)"
    )

    # === AUDIT ===
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False
    )

    # === INDEXES ===
    __table_args__ = (
        # Performance: filter by is_active
        Index('ix_category_alias_active', 'is_active'),
        # Performance: lookup by target (para reverse lookups)
        Index('ix_category_alias_target', 'target_category'),
    )

    def __repr__(self):
        return (
            f"<CategoryAlias(id={self.id}, "
            f"'{self.source_category}' → '{self.target_category}', "
            f"active={self.is_active})>"
        )

    def increment_usage(self) -> None:
        """Incrementar contador de uso y actualizar timestamp."""
        self.usage_count += 1
        self.last_used_at = datetime.now(timezone.utc)
