"""
Modelo para tracking del historial de sincronización del catálogo CIMA/nomenclator.
Registra cada operación de sincronización con su estado, duración y resultados.
"""

import enum
import uuid
from datetime import datetime, timezone

from sqlalchemy import Column, DateTime
from sqlalchemy import Enum as SQLEnum
from sqlalchemy import Float, Integer, Text
from sqlalchemy.dialects.postgresql import UUID

from app.models.base import Base


class SyncType(str, enum.Enum):
    """Tipos de sincronización del catálogo"""

    CIMA = "CIMA"
    NOMENCLATOR = "NOMENCLATOR"
    CIMAVET = "CIMAVET"  # Sprint 3 - Issue #354: CIMAVet integration


class SyncStatus(str, enum.Enum):
    """Estados posibles de la sincronización"""

    # IMPORTANTE: Valores en minúsculas para coincidir con enum PostgreSQL 'syncstatus'
    SUCCESS = "success"
    FAILURE = "failure"
    PARTIAL = "partial"  # Sincronización parcial, algunos registros fallaron
    IN_PROGRESS = "in_progress"  # Sprint 3 - Issue #354: Sync en progreso
    COMPLETED = "completed"  # Sprint 3 - Issue #354: Sync completado exitosamente
    FAILED = "failed"  # Sprint 3 - Issue #354: Sync fallido


class TriggerType(str, enum.Enum):
    """Origen del trigger de sincronización"""

    AUTOMATIC = "AUTOMATIC"  # Sincronización programada/automática
    MANUAL = "MANUAL"        # Sincronización manual iniciada por admin


class CatalogSyncHistory(Base):
    """
    Modelo para almacenar el historial de sincronizaciones del catálogo.

    Registra cada intento de sincronización con CIMA o nomenclator,
    incluyendo su estado, duración, número de registros actualizados
    y cualquier error que haya ocurrido.
    """

    __tablename__ = "catalog_sync_history"
    __table_args__ = {"extend_existing": True}  # Thread-safe para multi-worker

    # Identificador único
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Información de la sincronización
    sync_date = Column(
        DateTime(timezone=True),
        nullable=False,
        default=lambda: datetime.now(timezone.utc),
        index=True  # Índice para ordenamiento por fecha
    )
    sync_type = Column(
        SQLEnum(SyncType),
        nullable=False,
        index=True  # Índice para filtrado por tipo
    )
    status = Column(
        SQLEnum(
            SyncStatus,
            values_callable=lambda e: [x.value for x in e]  # Usar .value (minúsculas) para PostgreSQL
        ),
        nullable=False,
        index=True  # Índice para filtrado por estado
    )

    # Métricas de la sincronización
    records_updated = Column(Integer, nullable=False, default=0)
    duration_seconds = Column(Float, nullable=False, default=0.0)

    # Origen de la sincronización
    triggered_by = Column(
        SQLEnum(TriggerType),
        nullable=False,
        default=TriggerType.AUTOMATIC
    )

    # Información de error (nullable para sincronizaciones exitosas)
    error_message = Column(Text, nullable=True)

    # Timestamp de creación del registro
    created_at = Column(
        DateTime(timezone=True),
        nullable=False,
        default=lambda: datetime.now(timezone.utc)
    )

    def to_dict(self):
        """
        Convierte el modelo a diccionario para API responses.

        Returns:
            dict: Representación en diccionario del registro de sincronización
        """
        return {
            "sync_date": self.sync_date.isoformat() if self.sync_date else None,
            "sync_type": self.sync_type.value if self.sync_type else None,
            "status": self.status.value if self.status else None,
            "records_updated": self.records_updated,
            "duration_seconds": self.duration_seconds,
            "triggered_by": self.triggered_by.value if self.triggered_by else None,
            "error_message": self.error_message
        }

    def __repr__(self):
        return (
            f"<CatalogSyncHistory("
            f"sync_type={self.sync_type}, "
            f"status={self.status}, "
            f"records={self.records_updated}, "
            f"date={self.sync_date}"
            f")>"
        )
