# backend/app/models/brand_alias.py
"""
Modelo para almacenar aliases de marcas detectadas.

Issue #XXX: Sistema de corrección de marcas para BrandDetectionService.

Propósito:
    - Fusionar marcas: "oralkin" → "kin" (misma marca, diferentes nombres)
    - Excluir falsos positivos: "clorhexidina" → NULL (ingrediente, no marca)

Beneficios vs hardcoded:
    - Cambios sin deploys (gestionable desde Admin UI)
    - Audit trail (quién, cuándo, por qué)
    - Analytics de uso (usage_count, last_used_at)
    - Re-proceso de productos afectados
"""

from datetime import datetime, timezone
from enum import Enum

from sqlalchemy import (
    Boolean,
    Column,
    DateTime,
    Index,
    Integer,
    String,
    Enum as SQLEnum,
)

from app.database import Base


class BrandAliasAction(str, Enum):
    """Acción a realizar sobre la marca detectada."""

    ALIAS = "alias"  # Renombrar a target_brand
    EXCLUDE = "exclude"  # No es marca, excluir de detección


class BrandAlias(Base):
    """
    Alias de marca para normalización/exclusión en detección de marcas.

    Ejemplos:
        - action=ALIAS: "oralkin" → "kin" (fusionar marcas)
        - action=EXCLUDE: "clorhexidina" → NULL (no es marca, es ingrediente)

    Diferencias con CategoryAlias:
        - Campo `action` para distinguir alias vs exclusión
        - `target_brand` nullable (NULL cuando action=EXCLUDE)
        - Incluye `affected_count` para re-proceso
    """

    __tablename__ = "brand_aliases"

    id = Column(Integer, primary_key=True, index=True)

    # === ALIAS DATA ===
    # Lo que detecta BrandDetectionService
    source_brand = Column(
        String(100),
        nullable=False,
        unique=True,
        comment="Marca detectada (input a normalizar/excluir)",
    )

    # Lo que debe ser (NULL para exclusión)
    target_brand = Column(
        String(100),
        nullable=True,
        comment="Marca normalizada (output). NULL si action=EXCLUDE",
    )

    # Tipo de acción
    action = Column(
        SQLEnum(BrandAliasAction, name="brand_alias_action_enum"),
        nullable=False,
        default=BrandAliasAction.ALIAS,
        comment="Acción: 'alias' (renombrar) o 'exclude' (eliminar detección)",
    )

    # === STATUS ===
    is_active = Column(
        Boolean,
        nullable=False,
        default=True,
        comment="Solo activos se usan en detección",
    )

    # === METADATA ===
    reason = Column(
        String(500),
        nullable=True,
        comment="Razón del alias (ej: 'Es ingrediente, no marca')",
    )

    # === ANALYTICS ===
    usage_count = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Veces que se ha usado este alias en detección",
    )

    last_used_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Última vez que se usó (detectar obsoletos)",
    )

    # === AUDIT ===
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False,
    )

    # === INDEXES ===
    __table_args__ = (
        # Performance: filter by is_active
        Index("ix_brand_alias_active", "is_active"),
        # Performance: lookup by target (para reverse lookups)
        Index("ix_brand_alias_target", "target_brand"),
        # Performance: filter by action type
        Index("ix_brand_alias_action", "action"),
    )

    def __repr__(self):
        if self.action == BrandAliasAction.EXCLUDE:
            return (
                f"<BrandAlias(id={self.id}, "
                f"'{self.source_brand}' → EXCLUDE, "
                f"active={self.is_active})>"
            )
        return (
            f"<BrandAlias(id={self.id}, "
            f"'{self.source_brand}' → '{self.target_brand}', "
            f"active={self.is_active})>"
        )

    def increment_usage(self) -> None:
        """Incrementar contador de uso y actualizar timestamp."""
        self.usage_count += 1
        self.last_used_at = datetime.now(timezone.utc)
