# backend/app/models/base.py
"""
Base común para todos los modelos SQLAlchemy con thread-safe singleton pattern
"""
import threading

from app.database import Base

# Thread-safe singleton para prevenir conflictos SQLAlchemy MetaData
_models_registered = False
_models_lock = threading.Lock()


def safe_model_registry():
    """
    Thread-safe singleton pattern para registro de modelos SQLAlchemy.
    Previene "Table already defined for this MetaData instance" en deployments multi-instancia.

    NOTA IMPORTANTE (2025-09-07):
    - En Render se ejecutan múltiples workers que causan conflicto si los modelos se importan
      tanto aquí como en __init__.py
    - SOLUCIÓN: Los modelos YA están importados en __init__.py, NO importar aquí también
    - Este problema NO ocurre en desarrollo local porque solo hay 1 proceso/worker

    Returns:
        Base: SQLAlchemy declarative base
    """
    global _models_registered
    if _models_registered:
        return Base

    with _models_lock:
        # Double-check locking pattern
        if not _models_registered:
            # NO IMPORTAR MODELOS AQUÍ - ya están en __init__.py
            # La doble importación causa "Multiple classes found for path 'User'" en Render
            _models_registered = True

    return Base


# Re-exportar Base para que los modelos la usen
__all__ = ["Base", "safe_model_registry"]
