# backend/app/models/backup_metadata.py
import uuid

from sqlalchemy import BigInteger, Column, DateTime, ForeignKey, String
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.database import Base
from app.utils.datetime_utils import utc_now


class BackupMetadata(Base):
    """
    Modelo para metadatos de backups con verificación de integridad.

    Issue #348: Admin Panel Unificado - Tab Database - Backup Manager.

    Características:
    - SHA-256 hash: Detección de corrupción/modificación del archivo
    - HMAC signature: Autenticidad y no repudio (firmado con secret key)
    - Size tracking: Monitoreo de espacio de almacenamiento
    - Creator tracking: Auditoría de quién creó el backup (con soft FK)

    Seguridad:
    - Integridad: Hash SHA-256 verifica que el archivo no fue modificado
    - Autenticidad: HMAC firma verifica que el backup fue creado por el sistema
    - No repudio: HMAC signature prueba origen legítimo del backup

    GDPR Compliance:
    - Audit trail: Quién creó el backup y cuándo
    - Restoration capability: Recuperación completa de datos
    """

    __tablename__ = "backup_metadata"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Backup file information
    filename = Column(String(255), unique=True, nullable=False, index=True)
    size_bytes = Column(BigInteger, nullable=False)

    # Integrity verification
    sha256_hash = Column(String(64), nullable=False)  # SHA-256 produces 64 hex chars
    hmac_signature = Column(String(64), nullable=False)  # HMAC-SHA256 also 64 chars

    # Audit trail (soft FK - no CASCADE delete when user deleted)
    created_by_user_id = Column(
        UUID(as_uuid=True),
        ForeignKey("users.id", ondelete="SET NULL"),
        nullable=True,
        index=True
    )

    # Timestamps
    created_at = Column(DateTime(timezone=True), default=utc_now, index=True)

    # Relationships
    created_by = relationship("User", foreign_keys=[created_by_user_id])

    def __repr__(self):
        size_mb = self.size_bytes / (1024 * 1024)
        return f"<BackupMetadata {self.filename} ({size_mb:.2f} MB)>"

    def to_dict(self, include_signatures: bool = False):
        """
        Convierte el modelo a diccionario para APIs.

        Args:
            include_signatures: Si True, incluye hashes/signatures (solo para admins)

        Returns:
            Diccionario con metadatos del backup
        """
        data = {
            "id": str(self.id),
            "filename": self.filename,
            "size_bytes": self.size_bytes,
            "size_mb": round(self.size_bytes / (1024 * 1024), 2),
            "size_gb": round(self.size_bytes / (1024 * 1024 * 1024), 3),
            "created_by_user_id": str(self.created_by_user_id) if self.created_by_user_id else None,
            "created_at": self.created_at.isoformat() if self.created_at else None
        }

        # Solo incluir hashes/signatures para verificación de integridad
        if include_signatures:
            data.update({
                "sha256_hash": self.sha256_hash,
                "hmac_signature": self.hmac_signature
            })

        return data

    def verify_integrity(self, file_content: bytes, hmac_key: str) -> dict:
        """
        Verifica la integridad del backup comparando hash y signature.

        Args:
            file_content: Contenido del archivo backup en bytes
            hmac_key: Secret key para verificar HMAC signature

        Returns:
            {
                "valid": bool,
                "sha256_match": bool,
                "hmac_match": bool,
                "details": str
            }
        """
        import hashlib
        import hmac as hmac_lib

        # Calcular SHA-256 del archivo
        calculated_sha256 = hashlib.sha256(file_content).hexdigest()
        sha256_match = calculated_sha256 == self.sha256_hash

        # Calcular HMAC signature
        calculated_hmac = hmac_lib.new(
            hmac_key.encode(),
            file_content,
            hashlib.sha256
        ).hexdigest()
        hmac_match = calculated_hmac == self.hmac_signature

        # Resultado de verificación
        valid = sha256_match and hmac_match

        return {
            "valid": valid,
            "sha256_match": sha256_match,
            "hmac_match": hmac_match,
            "details": (
                "✓ Backup integrity verified" if valid else
                "✗ Backup integrity FAILED"
            ),
            "sha256": {
                "expected": self.sha256_hash,
                "calculated": calculated_sha256
            },
            "hmac": {
                "expected": self.hmac_signature,
                "calculated": calculated_hmac
            }
        }

    @classmethod
    def create_from_file(cls, filename: str, file_content: bytes, hmac_key: str, user_id: UUID = None):
        """
        Crea un registro de backup con hashes calculados.

        Args:
            filename: Nombre del archivo backup
            file_content: Contenido del archivo en bytes
            hmac_key: Secret key para generar HMAC signature
            user_id: ID del usuario que creó el backup (opcional)

        Returns:
            Instancia de BackupMetadata con hashes calculados
        """
        import hashlib
        import hmac as hmac_lib

        # Calcular hashes
        sha256_hash = hashlib.sha256(file_content).hexdigest()
        hmac_signature = hmac_lib.new(
            hmac_key.encode(),
            file_content,
            hashlib.sha256
        ).hexdigest()

        # Crear instancia
        return cls(
            filename=filename,
            size_bytes=len(file_content),
            sha256_hash=sha256_hash,
            hmac_signature=hmac_signature,
            created_by_user_id=user_id
        )
