# backend/app/models/audit_log.py
"""
Audit log model for tracking critical operations in the system.
"""

import enum
from uuid import uuid4

from sqlalchemy import JSON, Column, DateTime
from sqlalchemy import Enum as SQLEnum
from sqlalchemy import Index, String, Text
from sqlalchemy.dialects.postgresql import UUID as PGUUID

from app.database import Base
from app.utils.datetime_utils import utc_now


class AuditAction(enum.Enum):
    """Types of audit actions"""

    LOGIN = "login"
    LOGOUT = "logout"
    CREATE = "create"
    READ = "read"
    UPDATE = "update"
    DELETE = "delete"
    UPLOAD = "upload"
    EXPORT = "export"
    ADMIN_ACTION = "admin_action"
    SYSTEM_CHANGE = "system_change"


class AuditSeverity(enum.Enum):
    """Severity levels for audit logs"""

    INFO = "info"
    WARNING = "warning"
    CRITICAL = "critical"


class AuditLog(Base):
    """
    Model for audit trail of all critical operations.
    """

    __tablename__ = "audit_logs"
    __table_args__ = (
        # Composite indexes for common query patterns
        Index("ix_audit_user_timestamp", "user_id", "timestamp"),
        Index("ix_audit_severity_timestamp", "severity", "timestamp"),
        Index("ix_audit_action_timestamp", "action", "timestamp"),
        Index("ix_audit_success_timestamp", "success", "timestamp"),
        {"extend_existing": True},
    )

    id = Column(PGUUID(as_uuid=True), primary_key=True, default=uuid4)
    timestamp = Column(DateTime, nullable=False, default=utc_now, index=True)

    # User information
    user_id = Column(PGUUID(as_uuid=True), nullable=True)  # Nullable for system actions
    user_email = Column(String(255), nullable=True)
    user_role = Column(String(50), nullable=True)

    # Action details
    action = Column(SQLEnum(AuditAction), nullable=False, index=True)
    severity = Column(SQLEnum(AuditSeverity), nullable=False, default=AuditSeverity.INFO)
    resource_type = Column(String(100), nullable=True)  # e.g., "pharmacy", "sales_data", "user"
    resource_id = Column(String(255), nullable=True)  # ID of the affected resource

    # Request information
    method = Column(String(10), nullable=False)  # HTTP method
    endpoint = Column(String(500), nullable=False)  # API endpoint
    ip_address = Column(String(45), nullable=True)  # IPv4 or IPv6
    user_agent = Column(Text, nullable=True)

    # Operation details
    description = Column(Text, nullable=True)  # Human-readable description
    details = Column(JSON, nullable=True)  # Additional structured data

    # Result
    success = Column(String(10), nullable=False, default="success")  # "success" or "failure"
    status_code = Column(String(10), nullable=True)  # HTTP status code
    error_message = Column(Text, nullable=True)  # Error details if failed

    # Performance metrics
    duration_ms = Column(String(20), nullable=True)  # Request duration in milliseconds

    # Indexing
    created_at = Column(DateTime, nullable=False, default=utc_now, index=True)
    updated_at = Column(DateTime, nullable=False, default=utc_now, onupdate=utc_now)

    def __repr__(self):
        return f"<AuditLog {self.id}: {self.user_email} - {self.action.value} - {self.endpoint}>"

    def to_dict(self):
        """Convert audit log to dictionary for JSON serialization"""
        return {
            "id": str(self.id),
            "timestamp": self.timestamp.isoformat() if self.timestamp else None,
            "user_id": str(self.user_id) if self.user_id else None,
            "user_email": self.user_email,
            "user_role": self.user_role,
            "action": self.action.value if self.action else None,
            "severity": self.severity.value if self.severity else None,
            "resource_type": self.resource_type,
            "resource_id": self.resource_id,
            "method": self.method,
            "endpoint": self.endpoint,
            "ip_address": self.ip_address,
            "success": self.success,
            "status_code": self.status_code,
            "description": self.description,
            "details": self.details,
            "duration_ms": self.duration_ms,
        }
