# backend/app/models/action_tracking.py
"""
Modelos para el sistema ROI Tracker (Feedback Loop de Acciones).

Issue #514: Cerrar el círculo diagnóstico → acción → resultado.
Demostrar ROI de xFarma con datos reales.

Filosofía:
- ActionTracking: Registra acciones sugeridas por el sistema y su ejecución
- MonthlyROISummary: Agrega impacto mensual para mostrar valor de suscripción

Tipos de acción:
- LIQUIDATION: Liquidar stock muerto con descuento
- RESTOCK: Reponer urgente para evitar rotura
- PRICING: Ajustar precio/margen
- DIVERSIFY: Añadir nuevos SKUs a categoría

Estados:
- PENDING: Sugerida pero no accionada
- EXECUTED: Usuario marcó como ejecutada
- DISCARDED: Usuario descartó la sugerencia
- POSTPONED: Pospuesta para más adelante
"""

from datetime import datetime, timezone
from typing import Any, Dict, List, Optional
from uuid import uuid4

import enum

from sqlalchemy import (
    Column,
    Date,
    DateTime,
    Float,
    ForeignKey,
    Index,
    Integer,
    String,
    Text,
    UniqueConstraint,
)
from sqlalchemy.dialects.postgresql import JSONB, UUID
from sqlalchemy.orm import relationship

from app.database import Base


# =============================================================================
# ENUMS (usando String para evitar issues multi-worker)
# =============================================================================

class ActionType(str, enum.Enum):
    """Tipo de acción sugerida por el sistema."""
    LIQUIDATION = "liquidation"  # Liquidar stock con descuento
    RESTOCK = "restock"          # Reponer urgente
    PRICING = "pricing"          # Ajuste precio/margen
    DIVERSIFY = "diversify"      # Añadir SKUs a categoría


class ActionStatus(str, enum.Enum):
    """Estado de la acción en el flujo de trabajo."""
    PENDING = "pending"          # Sugerida, pendiente de acción
    EXECUTED = "executed"        # Ejecutada por el usuario
    DISCARDED = "discarded"      # Descartada permanentemente
    POSTPONED = "postponed"      # Pospuesta para fecha futura


# =============================================================================
# ACTION TRACKING MODEL
# =============================================================================

class ActionTracking(Base):
    """
    Registro de acciones sugeridas por el sistema y su seguimiento.

    Cada acción tiene un impacto esperado (calculado al sugerir) y un
    impacto real (calculado tras ejecución). La diferencia demuestra
    la precisión del sistema y el ROI para el usuario.

    Relacionado con InsightFeedback via insight_hash para trazabilidad.
    """
    __tablename__ = "action_trackings"

    # -------------------------------------------------------------------------
    # Primary key
    # -------------------------------------------------------------------------
    id = Column(
        UUID(as_uuid=True),
        primary_key=True,
        default=uuid4,
    )

    # -------------------------------------------------------------------------
    # Foreign key to pharmacy
    # -------------------------------------------------------------------------
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # -------------------------------------------------------------------------
    # Action details
    # -------------------------------------------------------------------------
    action_type = Column(
        String(20),
        nullable=False,
        comment="Tipo: liquidation, restock, pricing, diversify",
    )

    action_description = Column(
        Text,
        nullable=False,
        comment="Descripción legible de la acción sugerida",
    )

    affected_products = Column(
        JSONB,
        nullable=False,
        default=list,
        comment="Lista de productos afectados [{codigo_nacional, nombre, ...}]",
    )

    # -------------------------------------------------------------------------
    # Expected impact (calculado al crear la sugerencia)
    # -------------------------------------------------------------------------
    expected_impact_eur = Column(
        Float,
        nullable=False,
        default=0.0,
        comment="Impacto económico esperado en EUR",
    )

    expected_impact_description = Column(
        Text,
        nullable=True,
        comment="Explicación del cálculo del impacto esperado",
    )

    # -------------------------------------------------------------------------
    # Status and transitions
    # -------------------------------------------------------------------------
    status = Column(
        String(20),
        nullable=False,
        default=ActionStatus.PENDING.value,
        comment="Estado: pending, executed, discarded, postponed",
    )

    executed_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha/hora en que el usuario marcó como ejecutada",
    )

    discarded_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha/hora en que el usuario descartó la acción",
    )

    discard_reason = Column(
        Text,
        nullable=True,
        comment="Razón opcional por la que se descartó",
    )

    postponed_until = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha hasta la cual está pospuesta",
    )

    # -------------------------------------------------------------------------
    # Actual impact (calculado tras ejecución)
    # -------------------------------------------------------------------------
    actual_impact_eur = Column(
        Float,
        nullable=True,
        comment="Impacto real medido en EUR (null si no calculado aún)",
    )

    actual_impact_description = Column(
        Text,
        nullable=True,
        comment="Explicación del impacto real calculado",
    )

    impact_calculated_at = Column(
        DateTime(timezone=True),
        nullable=True,
        comment="Fecha/hora del último cálculo de impacto real",
    )

    # -------------------------------------------------------------------------
    # Source references
    # -------------------------------------------------------------------------
    insight_hash = Column(
        String(64),
        nullable=True,
        index=True,
        comment="Hash del insight que generó esta acción (link a InsightFeedback)",
    )

    # -------------------------------------------------------------------------
    # Timestamps
    # -------------------------------------------------------------------------
    created_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False,
    )

    updated_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        onupdate=lambda: datetime.now(timezone.utc),
        nullable=True,
    )

    # -------------------------------------------------------------------------
    # Relationships
    # -------------------------------------------------------------------------
    pharmacy = relationship("Pharmacy", lazy="selectin")

    # -------------------------------------------------------------------------
    # Indexes
    # -------------------------------------------------------------------------
    __table_args__ = (
        # Composite index for listing actions by pharmacy and status
        Index(
            "ix_action_tracking_pharmacy_status",
            "pharmacy_id",
            "status",
        ),
        # Index for finding actions by type
        Index(
            "ix_action_tracking_pharmacy_type",
            "pharmacy_id",
            "action_type",
        ),
        # Index for monthly aggregation
        Index(
            "ix_action_tracking_created_at",
            "created_at",
        ),
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        return (
            f"<ActionTracking(id={self.id}, "
            f"type={self.action_type}, "
            f"status={self.status}, "
            f"expected={self.expected_impact_eur}EUR)>"
        )

    # -------------------------------------------------------------------------
    # Properties
    # -------------------------------------------------------------------------
    @property
    def is_pending(self) -> bool:
        """True si la acción está pendiente."""
        return self.status == ActionStatus.PENDING.value

    @property
    def is_executed(self) -> bool:
        """True si la acción fue ejecutada."""
        return self.status == ActionStatus.EXECUTED.value

    @property
    def is_discarded(self) -> bool:
        """True si la acción fue descartada."""
        return self.status == ActionStatus.DISCARDED.value

    @property
    def is_postponed(self) -> bool:
        """True si la acción está pospuesta."""
        return self.status == ActionStatus.POSTPONED.value

    @property
    def is_postpone_expired(self) -> bool:
        """True si el periodo de postpone expiró."""
        if not self.is_postponed or self.postponed_until is None:
            return False
        return datetime.now(timezone.utc) > self.postponed_until

    @property
    def has_impact_calculated(self) -> bool:
        """True si se calculó el impacto real."""
        return self.actual_impact_eur is not None

    @property
    def impact_delta(self) -> Optional[float]:
        """Diferencia entre impacto real y esperado (positivo = mejor de esperado)."""
        if self.actual_impact_eur is None:
            return None
        return self.actual_impact_eur - self.expected_impact_eur

    @property
    def impact_accuracy_pct(self) -> Optional[float]:
        """Porcentaje de precisión del impacto esperado vs real."""
        if self.actual_impact_eur is None or self.expected_impact_eur == 0:
            return None
        return (self.actual_impact_eur / self.expected_impact_eur) * 100

    def to_dict(self) -> Dict[str, Any]:
        """Serializa a diccionario para respuestas API."""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "action_type": self.action_type,
            "action_description": self.action_description,
            "affected_products": self.affected_products,
            "expected_impact_eur": self.expected_impact_eur,
            "expected_impact_description": self.expected_impact_description,
            "status": self.status,
            "executed_at": self.executed_at.isoformat() if self.executed_at else None,
            "discarded_at": self.discarded_at.isoformat() if self.discarded_at else None,
            "discard_reason": self.discard_reason,
            "postponed_until": self.postponed_until.isoformat() if self.postponed_until else None,
            "actual_impact_eur": self.actual_impact_eur,
            "actual_impact_description": self.actual_impact_description,
            "impact_calculated_at": self.impact_calculated_at.isoformat() if self.impact_calculated_at else None,
            "insight_hash": self.insight_hash,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
            # Computed properties
            "is_pending": self.is_pending,
            "is_executed": self.is_executed,
            "has_impact_calculated": self.has_impact_calculated,
            "impact_delta": self.impact_delta,
            "impact_accuracy_pct": self.impact_accuracy_pct,
        }


# =============================================================================
# MONTHLY ROI SUMMARY MODEL
# =============================================================================

class MonthlyROISummary(Base):
    """
    Resumen mensual de ROI para una farmacia.

    Agrega todas las acciones ejecutadas en el mes y calcula
    el ROI neto comparando impacto real vs coste de suscripción.

    Se recalcula periódicamente o bajo demanda.
    """
    __tablename__ = "monthly_roi_summaries"

    # -------------------------------------------------------------------------
    # Primary key
    # -------------------------------------------------------------------------
    id = Column(
        UUID(as_uuid=True),
        primary_key=True,
        default=uuid4,
    )

    # -------------------------------------------------------------------------
    # Foreign key to pharmacy
    # -------------------------------------------------------------------------
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # -------------------------------------------------------------------------
    # Month identifier (primer día del mes)
    # -------------------------------------------------------------------------
    month = Column(
        Date,
        nullable=False,
        comment="Primer día del mes (ej: 2026-01-01 para enero 2026)",
    )

    # -------------------------------------------------------------------------
    # Action counts
    # -------------------------------------------------------------------------
    actions_suggested = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Total de acciones sugeridas en el mes",
    )

    actions_executed = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Acciones marcadas como ejecutadas",
    )

    actions_discarded = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Acciones descartadas por el usuario",
    )

    actions_postponed = Column(
        Integer,
        nullable=False,
        default=0,
        comment="Acciones pospuestas (pendientes de re-evaluación)",
    )

    # -------------------------------------------------------------------------
    # Impact totals
    # -------------------------------------------------------------------------
    total_expected_impact = Column(
        Float,
        nullable=False,
        default=0.0,
        comment="Suma del impacto esperado de acciones ejecutadas (EUR)",
    )

    total_actual_impact = Column(
        Float,
        nullable=False,
        default=0.0,
        comment="Suma del impacto real medido (EUR)",
    )

    # -------------------------------------------------------------------------
    # Breakdown by type
    # -------------------------------------------------------------------------
    impact_by_type = Column(
        JSONB,
        nullable=False,
        default=dict,
        comment='Impacto por tipo: {"liquidation": 420.0, "restock": 380.0, ...}',
    )

    # -------------------------------------------------------------------------
    # ROI calculation
    # -------------------------------------------------------------------------
    subscription_cost = Column(
        Float,
        nullable=False,
        default=99.0,
        comment="Coste de suscripción mensual (EUR)",
    )

    roi_percentage = Column(
        Float,
        nullable=True,
        comment="ROI = (total_actual_impact / subscription_cost) * 100",
    )

    # -------------------------------------------------------------------------
    # Timestamps
    # -------------------------------------------------------------------------
    calculated_at = Column(
        DateTime(timezone=True),
        default=lambda: datetime.now(timezone.utc),
        nullable=False,
        comment="Fecha/hora del último cálculo",
    )

    # -------------------------------------------------------------------------
    # Relationships
    # -------------------------------------------------------------------------
    pharmacy = relationship("Pharmacy", lazy="selectin")

    # -------------------------------------------------------------------------
    # Constraints and indexes
    # -------------------------------------------------------------------------
    __table_args__ = (
        # Unique constraint: one summary per pharmacy per month
        UniqueConstraint(
            "pharmacy_id",
            "month",
            name="uq_roi_summary_pharmacy_month",
        ),
        # Index for time-series queries
        Index(
            "ix_roi_summary_month",
            "month",
        ),
        {"extend_existing": True},
    )

    def __repr__(self) -> str:
        month_str = self.month.strftime("%Y-%m") if self.month else "?"
        roi_str = f"{self.roi_percentage:.1f}%" if self.roi_percentage is not None else "N/A"
        return (
            f"<MonthlyROISummary(id={self.id}, "
            f"month={month_str}, "
            f"roi={roi_str})>"
        )

    # -------------------------------------------------------------------------
    # Properties
    # -------------------------------------------------------------------------
    @property
    def execution_rate(self) -> float:
        """Porcentaje de acciones ejecutadas vs sugeridas."""
        if self.actions_suggested == 0:
            return 0.0
        return (self.actions_executed / self.actions_suggested) * 100

    @property
    def net_roi(self) -> float:
        """ROI neto en EUR (impacto - coste suscripción)."""
        return self.total_actual_impact - self.subscription_cost

    @property
    def month_str(self) -> str:
        """Mes en formato YYYY-MM."""
        return self.month.strftime("%Y-%m") if self.month else ""

    def to_dict(self) -> Dict[str, Any]:
        """Serializa a diccionario para respuestas API."""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id),
            "month": self.month_str,
            "actions_suggested": self.actions_suggested,
            "actions_executed": self.actions_executed,
            "actions_discarded": self.actions_discarded,
            "actions_postponed": self.actions_postponed,
            "execution_rate": round(self.execution_rate, 1),
            "total_expected_impact": round(self.total_expected_impact, 2),
            "total_actual_impact": round(self.total_actual_impact, 2),
            "impact_by_type": self.impact_by_type,
            "subscription_cost": self.subscription_cost,
            "roi_percentage": round(self.roi_percentage, 1) if self.roi_percentage else None,
            "net_roi": round(self.net_roi, 2),
            "calculated_at": self.calculated_at.isoformat() if self.calculated_at else None,
        }
