﻿"""
Migración para eliminar campos redundantes de la tabla sales_data
Fecha: 2025-08-30
Motivo: Eliminación de redundancia de datos - normalización 3NF estricta

CAMPOS ELIMINADOS:
- laboratory (ya está en ProductCatalog.laboratory)
- active_ingredient (ya está en ProductCatalog.active_ingredient)
- therapeutic_group (ya está en ProductCatalog.therapeutic_group)
- category (ya está en ProductCatalog y SalesEnrichment)
- product_type (ya está en SalesEnrichment.product_type)
- is_generic (ya está en ProductCatalog.is_generic)

Los datos de estos campos siguen estando disponibles a través de las relaciones:
SalesData -> SalesEnrichment -> ProductCatalog
"""

import logging
import os

from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker

from app.utils.datetime_utils import utc_now

# Configuración de logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def run_migration():
    """
    Ejecuta la migración para eliminar campos redundantes
    """

    # Obtener URL de la base de datos
    database_url = os.getenv(
        "DATABASE_URL",
        "postgresql://xfarma_user:xfarma_dev_2024@localhost:5432/xfarma_db",
    )

    engine = create_engine(database_url)
    SessionLocal = sessionmaker(bind=engine)

    logger.info("=== INICIANDO MIGRACIÓN: Eliminación campos redundantes sales_data ===")

    with engine.connect() as connection:
        # Iniciar transacción
        trans = connection.begin()

        try:
            # 1. Verificar que la tabla existe
            logger.info("1. Verificando existencia de tabla sales_data...")
            result = connection.execute(
                text(
                    """
                SELECT EXISTS (
                    SELECT FROM information_schema.tables
                    WHERE table_schema = 'public' AND table_name = 'sales_data'
                );
            """
                )
            )
            table_exists = result.scalar()

            if not table_exists:
                logger.warning("La tabla sales_data no existe. Migración omitida.")
                trans.rollback()
                return

            # 2. Verificar columnas existentes
            logger.info("2. Verificando columnas a eliminar...")
            columns_to_remove = [
                "laboratory",
                "active_ingredient",
                "therapeutic_group",
                "category",
                "product_type",
                "is_generic",
            ]

            existing_columns = []
            for column in columns_to_remove:
                result = connection.execute(
                    text(
                        f"""
                    SELECT EXISTS (
                        SELECT FROM information_schema.columns
                        WHERE table_name = 'sales_data' AND column_name = '{column}'
                    );
                """  # nosec B608 - column names are predefined constants, not user input
                    )
                )
                if result.scalar():
                    existing_columns.append(column)
                    logger.info(f"   ✓ Columna '{column}' encontrada")
                else:
                    logger.info(f"   - Columna '{column}' no existe (ya eliminada)")

            if not existing_columns:
                logger.info("No hay columnas para eliminar. Migración completada.")
                trans.commit()
                return

            # 3. Contar registros afectados (para logging)
            logger.info("3. Contando registros afectados...")
            result = connection.execute(text("SELECT COUNT(*) FROM sales_data;"))
            total_records = result.scalar()
            logger.info(f"   Total de registros en sales_data: {total_records}")

            # 4. Crear tabla de backup de metadatos (opcional)
            logger.info("4. Creando backup de metadatos...")
            backup_table_name = f"sales_data_redundant_backup_{utc_now().strftime('%Y%m%d_%H%M%S')}"

            # Solo crear backup si hay datos
            if total_records > 0:
                backup_columns = ", ".join(existing_columns)
                connection.execute(
                    text(
                        f"""
                    CREATE TABLE {backup_table_name} AS
                    SELECT id, {backup_columns}
                    FROM sales_data
                    WHERE {existing_columns[0]} IS NOT NULL;
                """  # nosec B608 - table and column names are generated programmatically from safe constants
                    )
                )

                result = connection.execute(
                    text(
                        f"SELECT COUNT(*) FROM {backup_table_name};"
                    )  # nosec B608 - table name is generated programmatically, not from user input
                )
                backup_count = result.scalar()
                logger.info(f"   ✓ Backup creado: {backup_table_name} ({backup_count} registros)")

            # 5. Eliminar columnas redundantes
            logger.info("5. Eliminando columnas redundantes...")
            for column in existing_columns:
                logger.info(f"   Eliminando columna: {column}")
                connection.execute(text(f"ALTER TABLE sales_data DROP COLUMN IF EXISTS {column};"))
                logger.info(f"   ✓ Columna '{column}' eliminada")

            # 6. Verificar que las columnas fueron eliminadas
            logger.info("6. Verificando eliminación...")
            for column in existing_columns:
                result = connection.execute(
                    text(
                        f"""
                    SELECT EXISTS (
                        SELECT FROM information_schema.columns
                        WHERE table_name = 'sales_data' AND column_name = '{column}'
                    );
                """  # nosec B608 - column names are predefined constants, not user input
                    )
                )
                if not result.scalar():
                    logger.info(f"   ✓ Columna '{column}' eliminada correctamente")
                else:
                    raise Exception(f"ERROR: Columna '{column}' todavía existe")

            # 7. Actualizar estadísticas de la tabla
            logger.info("7. Actualizando estadísticas de la tabla...")
            connection.execute(text("ANALYZE sales_data;"))
            logger.info("   ✓ Estadísticas actualizadas")

            # Confirmar transacción
            trans.commit()

            logger.info("=== MIGRACIÓN COMPLETADA EXITOSAMENTE ===")
            logger.info(f"Columnas eliminadas: {existing_columns}")
            logger.info(f"Registros afectados: {total_records}")
            if total_records > 0:
                logger.info(f"Backup disponible en: {backup_table_name}")
            logger.info("Los datos están disponibles a través de las relaciones SalesEnrichment -> ProductCatalog")

        except Exception as e:
            logger.error(f"ERROR durante la migración: {str(e)}")
            trans.rollback()
            raise e


def rollback_migration():
    """
    Rollback de la migración (solo si hay backup disponible)
    ADVERTENCIA: Solo usar en emergencias
    """
    logger.warning("=== ROLLBACK DE MIGRACIÓN NO IMPLEMENTADO ===")
    logger.warning("Los campos eliminados contenían datos redundantes.")
    logger.warning("Los datos están disponibles a través de las relaciones.")
    logger.warning("Si necesitas restaurar datos específicos, contacta al equipo de desarrollo.")


if __name__ == "__main__":
    import sys

    if len(sys.argv) > 1 and sys.argv[1] == "--rollback":
        rollback_migration()
    else:
        run_migration()
