# backend/app/middleware/versioning.py
"""
Middleware de versionado de API para xFarma.

Soporta múltiples estrategias de versionado:
1. Header: API-Version: v1
2. Query parameter: ?version=v1
3. Path prefix: /api/v1/...
4. Content-Type: application/vnd.xfarma.v1+json

Configuración por defecto: v1
"""

import re
from typing import Optional

import structlog
from fastapi import Request, Response
from fastapi.responses import JSONResponse
from starlette.middleware.base import BaseHTTPMiddleware

logger = structlog.get_logger(__name__)


class APIVersioningMiddleware(BaseHTTPMiddleware):
    """
    Middleware que maneja el versionado de la API xFarma.

    Estrategias de versionado soportadas:
    1. Header: API-Version: v1
    2. Query param: ?version=v1
    3. Path: /api/v1/endpoint
    4. Content-Type: application/vnd.xfarma.v1+json
    """

    SUPPORTED_VERSIONS = {"v1", "v2"}  # Versiones soportadas
    DEFAULT_VERSION = "v1"

    # Endpoints que requieren versionado específico
    VERSION_SPECIFIC_ENDPOINTS = {
        "v2": {
            # Endpoints solo disponibles en v2
            "/sales/analytics/advanced": "v2",
            "/catalog/batch-operations": "v2",
            "/enrichment/batch-process": "v2",
        }
    }

    def __init__(self, app, default_version: str = "v1"):
        super().__init__(app)
        self.default_version = default_version
        logger.info(
            "api_versioning.initialized",
            default_version=default_version,
            supported_versions=list(self.SUPPORTED_VERSIONS),
        )

    async def dispatch(self, request: Request, call_next):
        """Procesa la request para determinar y validar la versión de API"""

        # 1. Detectar versión solicitada
        requested_version = await self._detect_api_version(request)

        # 2. Validar versión
        if requested_version not in self.SUPPORTED_VERSIONS:
            logger.warning(
                "api_versioning.unsupported_version",
                requested=requested_version,
                supported=list(self.SUPPORTED_VERSIONS),
                path=request.url.path,
            )
            return self._version_error_response(requested_version)

        # 3. Validar compatibilidad del endpoint
        compatibility_error = self._check_endpoint_compatibility(request.url.path, requested_version)
        if compatibility_error:
            return compatibility_error

        # 4. Agregar versión al contexto de la request
        request.state.api_version = requested_version

        # 5. Log de la versión detectada
        logger.info(
            "api_versioning.request_processed",
            version=requested_version,
            path=request.url.path,
            method=request.method,
        )

        # 6. Procesar request
        response = await call_next(request)

        # 7. Agregar headers de versión a la response
        response.headers["API-Version"] = requested_version
        response.headers["API-Supported-Versions"] = ",".join(self.SUPPORTED_VERSIONS)

        return response

    async def _detect_api_version(self, request: Request) -> str:
        """
        Detecta la versión de API solicitada usando múltiples estrategias.
        Orden de precedencia: Header > Query > Path > Content-Type > Default
        """

        # 1. Header: API-Version
        header_version = request.headers.get("API-Version")
        if header_version:
            logger.debug("api_versioning.detected_from_header", version=header_version)
            return header_version

        # 2. Query parameter: ?version=v1
        query_version = request.query_params.get("version")
        if query_version:
            logger.debug("api_versioning.detected_from_query", version=query_version)
            return query_version

        # 3. Path prefix: /api/v1/...
        path_version = self._extract_version_from_path(request.url.path)
        if path_version:
            logger.debug("api_versioning.detected_from_path", version=path_version)
            return path_version

        # 4. Content-Type: application/vnd.xfarma.v1+json
        content_type_version = self._extract_version_from_content_type(request.headers.get("content-type", ""))
        if content_type_version:
            logger.debug(
                "api_versioning.detected_from_content_type",
                version=content_type_version,
            )
            return content_type_version

        # 5. Default version
        logger.debug("api_versioning.using_default", version=self.default_version)
        return self.default_version

    def _extract_version_from_path(self, path: str) -> Optional[str]:
        """Extrae la versión del path: /api/v1/... -> v1"""
        # Patrón: /api/v{número}/...
        match = re.match(r"^/api/(v\d+)/", path)
        return match.group(1) if match else None

    def _extract_version_from_content_type(self, content_type: str) -> Optional[str]:
        """Extrae versión del Content-Type: application/vnd.xfarma.v1+json -> v1"""
        # Patrón: application/vnd.xfarma.v{número}+json
        match = re.search(r"application/vnd\.xfarma\.(v\d+)\+json", content_type)
        return match.group(1) if match else None

    def _check_endpoint_compatibility(self, path: str, version: str) -> Optional[Response]:
        """Verifica si el endpoint es compatible con la versión solicitada"""

        # Limpiar path para comparación (remover prefijo de versión si existe)
        clean_path = re.sub(r"^/api/v\d+", "", path)

        # Verificar endpoints específicos de versión
        for required_version, endpoints in self.VERSION_SPECIFIC_ENDPOINTS.items():
            if clean_path in endpoints:
                if version != required_version:
                    logger.warning(
                        "api_versioning.incompatible_endpoint",
                        path=clean_path,
                        requested_version=version,
                        required_version=required_version,
                    )
                    return self._incompatibility_error_response(clean_path, version, required_version)

        return None

    def _version_error_response(self, requested_version: str) -> JSONResponse:
        """Response para versión no soportada"""
        return JSONResponse(
            status_code=400,
            content={
                "error": "unsupported_api_version",
                "message": f"La versión de API '{requested_version}' no está soportada",
                "requested_version": requested_version,
                "supported_versions": list(self.SUPPORTED_VERSIONS),
                "default_version": self.default_version,
                "versioning_help": {
                    "header": "API-Version: v1",
                    "query": "?version=v1",
                    "path": "/api/v1/endpoint",
                    "content_type": "application/vnd.xfarma.v1+json",
                },
            },
            headers={
                "API-Supported-Versions": ",".join(self.SUPPORTED_VERSIONS),
                "API-Default-Version": self.default_version,
            },
        )

    def _incompatibility_error_response(self, path: str, requested: str, required: str) -> JSONResponse:
        """Response para endpoint incompatible con la versión"""
        return JSONResponse(
            status_code=400,
            content={
                "error": "endpoint_version_incompatible",
                "message": f"El endpoint '{path}' requiere la versión {required}",
                "endpoint": path,
                "requested_version": requested,
                "required_version": required,
                "suggestion": f"Use API-Version: {required} o ?version={required}",
            },
            headers={
                "API-Required-Version": required,
                "API-Supported-Versions": ",".join(self.SUPPORTED_VERSIONS),
            },
        )


# Utilidades para obtener versión en los endpoints
def get_api_version(request: Request) -> str:
    """
    Obtiene la versión de API de la request.
    Para usar en endpoints: version = get_api_version(request)
    """
    return getattr(request.state, "api_version", APIVersioningMiddleware.DEFAULT_VERSION)


def _version_to_number(version: str) -> int:
    """Convierte versión (v1, v2) a número para comparación"""
    if version.startswith("v"):
        try:
            return int(version[1:])
        except ValueError:
            return 0
    return 0


def requires_version(min_version: str):
    """
    Decorador para endpoints que requieren una versión mínima.

    Uso:
    @requires_version("v2")
    async def advanced_endpoint(request: Request):
        pass
    """

    def decorator(func):
        import functools


        @functools.wraps(func)
        async def wrapper(*args, **kwargs):
            # Buscar el Request en args y kwargs
            request = None

            # Buscar en argumentos posicionales
            for arg in args:
                if isinstance(arg, Request):
                    request = arg
                    break

            # Buscar en argumentos con nombre
            if request is None:
                for key, value in kwargs.items():
                    if isinstance(value, Request):
                        request = value
                        break

            # Si aún no encontramos Request, intentar con el primer argumento
            # que podría ser un Request sin tipo explícito
            if request is None and args:
                if hasattr(args[0], "state") and hasattr(args[0], "headers"):
                    request = args[0]

            if request is None:
                # Si no encontramos Request, ejecutar sin validación
                return await func(*args, **kwargs)

            current_version = get_api_version(request)
            current_num = _version_to_number(current_version)
            min_num = _version_to_number(min_version)

            if current_num < min_num:
                return JSONResponse(
                    status_code=400,
                    content={
                        "error": "version_too_old",
                        "message": f"Este endpoint requiere {min_version} o superior",
                        "current_version": current_version,
                        "minimum_version": min_version,
                    },
                )
            return await func(*args, **kwargs)

        return wrapper

    return decorator
