# backend/app/middleware/logging_middleware.py
"""
Middleware de logging optimizado para xFarma
Reduce ruido de logs en producción filtrando requests rutinarios manteniendo capacidad de debug
"""

import time
from typing import Set

import structlog
from fastapi import Request

logger = structlog.get_logger(__name__)


class LoggingConfig:
    """Configuración avanzada de logging para diferentes entornos"""

    # Paths que NO se loggean en producción (ruido excesivo)
    PRODUCTION_FILTERED_PATHS: Set[str] = {
        "/health",
        "/health/simple",
        "/_dash-update-component",
        "/_dash-layout",
        "/_dash-dependencies",
        "/_favicon.ico",
        "/assets/",
        "/static/",
    }

    # Paths que se loggean con nivel DEBUG en producción
    PRODUCTION_DEBUG_PATHS: Set[str] = {"/api/info", "/api/system/status", "/api/v1/system/status-unified"}

    # Paths que SIEMPRE se loggean (críticos para debugging)
    ALWAYS_LOG_PATHS: Set[str] = {
        "/api/v1/upload",
        "/api/v1/sales",
        "/api/v1/system/sync",
        "/api/admin",
        "/api/v1/partner-analysis",
        "/api/v1/generic-analysis",
    }

    # Status codes que siempre se loggean independientemente del path
    ALWAYS_LOG_STATUS_CODES: Set[int] = {400, 401, 403, 404, 422, 429, 500, 502, 503, 504}

    # Requests lentos que siempre se loggean (ms)
    SLOW_REQUEST_THRESHOLD_MS = 2000  # 2 segundos

    # User-Agents de health checks de servicios conocidos
    HEALTH_CHECK_USER_AGENTS: Set[str] = {"Render-HealthCheck", "kube-probe", "ELB-HealthChecker", "GoogleHC"}


class SmartLoggingMiddleware:
    """
    Middleware inteligente que filtra logs según contexto y entorno
    Mantiene observabilidad crítica mientras reduce ruido operacional
    """

    def __init__(self, environment: str = "development"):
        self.environment = environment
        self.is_production = environment == "production"
        self.request_count = 0
        self.last_stats_log = time.time()
        self.stats_interval = 300  # 5 minutos

        # Contadores para estadísticas
        self.stats = {
            "total_requests": 0,
            "filtered_requests": 0,
            "health_checks": 0,
            "dash_updates": 0,
            "slow_requests": 0,
            "error_requests": 0,
        }

    def should_log_request(self, request: Request, response_time_ms: float, status_code: int) -> tuple[bool, str]:
        """
        Determinar si un request debe ser loggeado y con qué nivel

        Returns:
            tuple[bool, str]: (should_log, log_level)
        """
        path = request.url.path
        user_agent = request.headers.get("user-agent", "")

        # SIEMPRE loggear si hay errores
        if status_code in LoggingConfig.ALWAYS_LOG_STATUS_CODES:
            self.stats["error_requests"] += 1
            return True, "ERROR"

        # SIEMPRE loggear si es request lento
        if response_time_ms >= LoggingConfig.SLOW_REQUEST_THRESHOLD_MS:
            self.stats["slow_requests"] += 1
            return True, "WARNING"

        # SIEMPRE loggear paths críticos
        if any(critical_path in path for critical_path in LoggingConfig.ALWAYS_LOG_PATHS):
            return True, "INFO"

        # En desarrollo, loggear todo
        if not self.is_production:
            return True, "INFO"

        # En producción, aplicar filtros inteligentes

        # Filtrar health checks de servicios conocidos
        if any(hc_agent in user_agent for hc_agent in LoggingConfig.HEALTH_CHECK_USER_AGENTS):
            self.stats["health_checks"] += 1
            return False, "DEBUG"

        # Filtrar paths rutinarios
        if any(filtered_path in path for filtered_path in LoggingConfig.PRODUCTION_FILTERED_PATHS):
            self.stats["filtered_requests"] += 1
            if path.startswith("/_dash-update-component"):
                self.stats["dash_updates"] += 1
            return False, "DEBUG"

        # Loggear con DEBUG paths informativos en producción
        if any(debug_path in path for debug_path in LoggingConfig.PRODUCTION_DEBUG_PATHS):
            return True, "DEBUG"

        # Por defecto, loggear requests normales en producción
        return True, "INFO"

    def log_periodic_stats(self):
        """Log estadísticas periódicas del middleware"""
        now = time.time()
        if now - self.last_stats_log >= self.stats_interval:
            total = self.stats["total_requests"]
            filtered = self.stats["filtered_requests"]
            filter_percentage = (filtered / total * 100) if total > 0 else 0

            logger.info(
                "logging.middleware.stats",
                total_requests=total,
                filtered_requests=filtered,
                filter_percentage=f"{filter_percentage:.1f}%",
                health_checks=self.stats["health_checks"],
                dash_updates=self.stats["dash_updates"],
                slow_requests=self.stats["slow_requests"],
                error_requests=self.stats["error_requests"],
                interval_minutes=self.stats_interval // 60,
            )

            self.last_stats_log = now

    async def __call__(self, request: Request, call_next):
        """Middleware principal de logging inteligente"""
        start_time = time.time()
        self.stats["total_requests"] += 1

        # Ejecutar request
        try:
            response = await call_next(request)
        except Exception as e:
            # Loggear siempre excepciones
            duration_ms = (time.time() - start_time) * 1000
            logger.error(
                "request.exception",
                path=request.url.path,
                method=request.method,
                duration_ms=round(duration_ms, 2),
                exception=str(e),
                client_ip=request.client.host if request.client else "unknown",
            )
            raise

        # Calcular tiempo de respuesta
        duration_ms = (time.time() - start_time) * 1000

        # Determinar si loggear
        should_log, log_level = self.should_log_request(request, duration_ms, response.status_code)

        if should_log:
            # Preparar datos de log
            log_data = {
                "path": request.url.path,
                "method": request.method,
                "status_code": response.status_code,
                "duration_ms": round(duration_ms, 2),
            }

            # Agregar IP solo si no es health check interno
            if not any(filtered_path in request.url.path for filtered_path in LoggingConfig.PRODUCTION_FILTERED_PATHS):
                log_data["client_ip"] = request.client.host if request.client else "unknown"

            # Agregar query params solo para requests importantes
            if request.url.query and any(
                important_path in request.url.path for important_path in LoggingConfig.ALWAYS_LOG_PATHS
            ):
                log_data["query"] = request.url.query

            # Log según nivel determinado
            if log_level == "ERROR":
                logger.error("request.completed", **log_data)
            elif log_level == "WARNING":
                logger.warning("request.slow", **log_data)
            elif log_level == "DEBUG":
                logger.debug("request.debug", **log_data)
            else:
                logger.info("request.completed", **log_data)

        # Log estadísticas periódicas
        self.log_periodic_stats()

        return response


# Factory para crear middleware configurado
def create_smart_logging_middleware(environment: str = "development") -> SmartLoggingMiddleware:
    """
    Factory para crear middleware de logging configurado según entorno

    Args:
        environment: "development", "production", etc.

    Returns:
        SmartLoggingMiddleware: Middleware configurado
    """
    return SmartLoggingMiddleware(environment=environment)
