# backend/app/middleware/license_guard.py
"""
License Guard Middleware for kaiFarma Local (Pivot 2026).

Enforces read-only mode when license kill-switch is active.
Blocks all write operations (POST, PUT, DELETE, PATCH) except for
license-related endpoints.

This provides a single point of enforcement instead of checking
read_only_mode in every endpoint.

Usage:
    In main.py:
        from app.middleware.license_guard import LicenseGuardMiddleware

        if LocalSecurityManager.is_local_mode():
            app.add_middleware(LicenseGuardMiddleware)
"""

import logging
from typing import Set

from fastapi import Request
from starlette.middleware.base import BaseHTTPMiddleware
from starlette.responses import JSONResponse, Response

logger = logging.getLogger(__name__)


class LicenseGuardMiddleware(BaseHTTPMiddleware):
    """
    Middleware that blocks write operations when license kill-switch is active.

    When app.state.read_only_mode is True:
    - GET, OPTIONS, HEAD: Allowed (read-only operations)
    - POST, PUT, DELETE, PATCH: Blocked with 402 Payment Required

    Exceptions (always allowed):
    - /api/v1/license/* - License status and revalidation
    - /api/v1/auth/* - Authentication operations
    - /health - Health checks
    """

    # Paths allowed even in read-only mode
    ALLOWED_PATHS: Set[str] = {
        "/health",
        "/metrics",
        "/api/v1/license",
        "/api/v1/auth",
    }

    # Read-only HTTP methods (always allowed)
    READ_ONLY_METHODS: Set[str] = {"GET", "OPTIONS", "HEAD"}

    async def dispatch(self, request: Request, call_next) -> Response:
        """Process each request through license guard logic."""
        # Check if kill-switch is active
        read_only_mode = getattr(request.app.state, "read_only_mode", False)

        if not read_only_mode:
            # Normal operation - allow all requests
            return await call_next(request)

        # Kill-switch is active
        path = request.url.path
        method = request.method

        # Allow read-only methods
        if method in self.READ_ONLY_METHODS:
            return await call_next(request)

        # Allow excluded paths (auth, license)
        if self._is_allowed_path(path):
            return await call_next(request)

        # Block write operations
        logger.warning(
            "[LICENSE_GUARD] Blocked write operation in read-only mode",
            extra={"method": method, "path": path},
        )

        return JSONResponse(
            status_code=402,  # Payment Required
            content={
                "detail": "Licencia expirada. Modo solo lectura activado.",
                "error_code": "LICENSE_EXPIRED",
                "message": (
                    "Su licencia ha expirado y el periodo de gracia ha finalizado. "
                    "Por favor, contacte con soporte para renovar su licencia."
                ),
                "actions": [
                    "Contacte con soporte: support@kaifarma.es",
                    "Conecte a internet para revalidar la licencia",
                ],
            },
        )

    def _is_allowed_path(self, path: str) -> bool:
        """Check if path is allowed even in read-only mode."""
        return any(path.startswith(allowed) for allowed in self.ALLOWED_PATHS)
