# backend/app/middleware/destructive_operations_logger.py
"""
Middleware para loggear operaciones destructivas que pueden eliminar datos.

Implementado para Issue #[INVESTIGACIÓN]: Pérdida de datos usuario dgruiz@gmail.com
Fecha: 2025-10-09

Este middleware registra todas las operaciones que pueden destruir datos en:
1. Logs estructurados (structlog)
2. Archivo separado para auditoría (/app/logs/destructive_operations.log)
3. Base de datos (tabla audit_logs)
"""
import os
from datetime import datetime, timezone
from pathlib import Path
from typing import Any, Dict, Optional

import structlog

logger = structlog.get_logger(__name__)


class AuditLoggingError(Exception):
    """Excepción cuando falla el logging de auditoría en operaciones CRÍTICAS"""

    pass


# Directorio para logs de auditoría
AUDIT_LOG_DIR = Path("/app/logs")
DESTRUCTIVE_OPS_LOG = AUDIT_LOG_DIR / "destructive_operations.log"


def ensure_audit_log_directory():
    """
    Crear directorio de logs de auditoría si no existe.

    Thread-safe para entornos multi-worker (Render).
    """
    try:
        # Crear directorio (exist_ok=True es thread-safe)
        AUDIT_LOG_DIR.mkdir(parents=True, exist_ok=True)

        # Solo cambiar permisos si necesario (evitar race condition)
        try:
            current_mode = AUDIT_LOG_DIR.stat().st_mode & 0o777
            if current_mode != 0o755:
                AUDIT_LOG_DIR.chmod(0o755)
        except (PermissionError, OSError):
            # Otro worker ya configuró permisos o no tenemos permisos
            # No es crítico, continuar
            pass

        # Validar que el directorio es escribible
        test_file = AUDIT_LOG_DIR / f".write_test_{os.getpid()}"
        try:
            test_file.touch()
            test_file.unlink()
        except (PermissionError, OSError) as e:
            logger.error("audit.directory.not_writable", dir=str(AUDIT_LOG_DIR), error=str(e))
            raise PermissionError(f"Directorio de audit logs no es escribible: {AUDIT_LOG_DIR}")

    except PermissionError:
        # Re-lanzar PermissionError (es crítico)
        raise
    except Exception as e:
        logger.error("audit.directory.creation_failed", error=str(e))
        # No lanzar excepción para permitir graceful degradation
        # (logs irán a structlog y BD aunque falle el archivo)


def log_destructive_operation(
    operation: str,
    user: str,
    details: Optional[Dict[str, Any]] = None,
    severity: str = "CRITICAL",
    success: bool = True,
    error_message: Optional[str] = None,
) -> None:
    """
    Registrar operación destructiva en múltiples destinos.

    Args:
        operation: Nombre de la operación (ej: 'alembic_downgrade', 'docker_volume_rm')
        user: Usuario que ejecutó la operación (email o 'system')
        details: Detalles adicionales de la operación
        severity: Nivel de severidad ('CRITICAL', 'WARNING', 'INFO')
        success: Si la operación fue exitosa
        error_message: Mensaje de error si falló
    """
    timestamp = datetime.now(timezone.utc).isoformat()
    details = details or {}

    # Información de contexto
    log_entry = {
        "timestamp": timestamp,
        "operation": operation,
        "user": user,
        "severity": severity,
        "success": success,
        "error_message": error_message,
        "details": details,
        "environment": os.getenv("ENVIRONMENT", "unknown"),
        "hostname": os.getenv("HOSTNAME", "unknown"),
    }

    # 1. Log estructurado con structlog
    if severity == "CRITICAL":
        logger.critical("destructive.operation.executed", **log_entry)
    elif severity == "WARNING":
        logger.warning("destructive.operation.attempted", **log_entry)
    else:
        logger.info("destructive.operation.logged", **log_entry)

    # 2. Escribir en archivo separado para auditoría
    try:
        ensure_audit_log_directory()

        with open(DESTRUCTIVE_OPS_LOG, "a", encoding="utf-8") as f:
            # Formato: timestamp | operation | user | success | details
            details_str = str(details).replace("\n", " ")
            log_line = (
                f"{timestamp} | "
                f"{severity} | "
                f"{operation} | "
                f"{user} | "
                f"{'SUCCESS' if success else 'FAILED'} | "
                f"{error_message or ''} | "
                f"{details_str}\n"
            )
            f.write(log_line)

    except Exception as e:
        logger.error("audit.file.write_failed", error=str(e), log_file=str(DESTRUCTIVE_OPS_LOG))

    # 3. Registrar en base de datos (audit_logs table)
    # FIX Issue #4: Retry logic para errores transitorios
    db_logging_success = False
    db_error = None
    max_retries = 3
    retry_delay = 1  # segundos

    for attempt in range(max_retries):
        try:
            import time

            from sqlalchemy.exc import IntegrityError, OperationalError

            from app.database import SessionLocal
            from app.models.audit_log import AuditLog

            db = SessionLocal()
            try:
                audit_entry = AuditLog(
                    user_email=user,
                    action=operation,
                    severity=severity.lower(),
                    description=f"Destructive operation: {operation}",
                    details=details,
                    success=success,
                    error_message=error_message,
                )
                db.add(audit_entry)
                db.commit()
                db_logging_success = True
                break  # Éxito, salir del loop

            except OperationalError as e:
                # Error transitorio (conexión, lock, etc.) - reintentar
                db_error = e
                db.rollback()
                logger.warning(
                    "audit.database.transient_error", error=str(e), attempt=attempt + 1, max_retries=max_retries
                )

                if attempt < max_retries - 1:
                    time.sleep(retry_delay * (attempt + 1))  # Backoff incremental
                    continue
                else:
                    # Último intento falló
                    raise

            except IntegrityError as e:
                # Error permanente (constraint violation) - no reintentar
                db_error = e
                db.rollback()
                logger.error("audit.database.integrity_error", error=str(e))
                raise

            finally:
                db.close()

        except Exception as e:
            db_error = e
            logger.error("audit.database.write_failed", error=str(e), attempt=attempt + 1)

            # Solo lanzar si es último intento
            if attempt == max_retries - 1:
                # Si es operación CRÍTICA y falló el logging en BD, DEBE fallar
                if severity == "CRITICAL":
                    raise AuditLoggingError(
                        f"CRÍTICO: No se pudo registrar operación destructiva '{operation}' "
                        f"en base de datos de auditoría tras {max_retries} intentos. "
                        f"Operación abortada por seguridad. Error: {str(e)}"
                    )
                break


def log_alembic_downgrade(revision: str, user: str, confirmed: bool = False) -> None:
    """Log específico para downgrade de Alembic"""
    log_destructive_operation(
        operation="alembic_downgrade",
        user=user,
        details={
            "revision": revision,
            "confirmed": confirmed,
            "action": "drop_all_tables",
        },
        severity="CRITICAL",
    )


def log_docker_volume_rm(volume_name: str, user: str) -> None:
    """Log específico para eliminación de volumen Docker"""
    log_destructive_operation(
        operation="docker_volume_rm",
        user=user,
        details={
            "volume_name": volume_name,
            "data_loss": "permanent",
        },
        severity="CRITICAL",
    )


def log_truncate_table(table_name: str, user: str, cascade: bool = False) -> None:
    """Log específico para TRUNCATE TABLE"""
    log_destructive_operation(
        operation="truncate_table",
        user=user,
        details={
            "table_name": table_name,
            "cascade": cascade,
        },
        severity="CRITICAL",
    )


def log_reset_script(
    script_name: str, user: str, backup_created: bool = False, backup_file: Optional[str] = None
) -> None:
    """Log específico para scripts reset.ps1, clean.ps1"""
    log_destructive_operation(
        operation="reset_script",
        user=user,
        details={
            "script_name": script_name,
            "backup_created": backup_created,
            "backup_file": backup_file,
        },
        severity="CRITICAL",
    )


def get_audit_log_summary(days: int = 7) -> Dict[str, Any]:
    """
    Obtener resumen de operaciones destructivas de los últimos N días.

    Args:
        days: Número de días hacia atrás

    Returns:
        Dict con estadísticas de operaciones destructivas
    """
    try:
        from datetime import timedelta

        from app.database import SessionLocal
        from app.models.audit_log import AuditLog

        db = SessionLocal()
        try:
            since = datetime.now(timezone.utc) - timedelta(days=days)

            destructive_ops = (
                db.query(AuditLog)
                .filter(
                    AuditLog.action.in_(
                        [
                            "alembic_downgrade",
                            "docker_volume_rm",
                            "truncate_table",
                            "reset_script",
                        ]
                    )
                )
                .filter(AuditLog.created_at >= since)
                .all()
            )

            return {
                "total_operations": len(destructive_ops),
                "by_operation": {},
                "by_user": {},
                "failed_operations": len([op for op in destructive_ops if not op.success]),
                "period_days": days,
            }

        finally:
            db.close()

    except Exception as e:
        logger.error("audit.summary.failed", error=str(e))
        return {"error": str(e)}
