# backend/app/middleware/autolock_middleware.py
"""
Auto-Lock Middleware for kaiFarma Local (Pivot 2026).

Monitors user activity and triggers auto-lock after inactivity period.
Only active in local mode (KAIFARMA_LOCAL=true).
"""

import logging
from typing import Set

from fastapi import Request
from starlette.middleware.base import BaseHTTPMiddleware
from starlette.responses import Response

from app.core.security_local.local_state import security_manager

logger = logging.getLogger(__name__)


class AutoLockMiddleware(BaseHTTPMiddleware):
    """
    Middleware that:
    1. Checks if terminal should auto-lock due to inactivity
    2. Refreshes activity timestamp on user interactions
    3. Skips certain paths (health checks, static files, auth status)
    """

    # Paths that don't reset the activity timer or require auth
    EXCLUDED_PATHS: Set[str] = {
        "/health",
        "/metrics",
        "/static",
        "/_dash",
        "/api/v1/auth/status",  # Allow status checks without refreshing timer
        "/api/v1/auth/unlock",  # Allow unlock attempts
        "/api/v1/auth/lock",  # Allow manual lock
    }

    async def dispatch(self, request: Request, call_next) -> Response:
        """Process each request through the auto-lock logic."""
        path = request.url.path

        # Skip excluded paths
        if self._is_excluded_path(path):
            return await call_next(request)

        # Check if should auto-lock (idempotent - safe to call often)
        was_locked = security_manager.check_auto_lock()
        if was_locked:
            logger.info("[AUTOLOCK] Terminal locked due to inactivity")

        # Refresh activity timestamp if session is unlocked
        session = security_manager.get_session()
        if session and session.is_unlocked:
            security_manager.refresh_activity()

        # Continue with the request
        return await call_next(request)

    def _is_excluded_path(self, path: str) -> bool:
        """Check if path should be excluded from activity tracking."""
        return any(path.startswith(excluded) for excluded in self.EXCLUDED_PATHS)
