﻿"""
Middleware para estandarización de API.
Añade timestamps, versionado y otros estándares a todas las respuestas.
"""

import json
import time

import structlog
from fastapi import Request
from fastapi.responses import JSONResponse
from starlette.middleware.base import BaseHTTPMiddleware

from ..utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


class APIStandardsMiddleware(BaseHTTPMiddleware):
    """
    Middleware que añade estándares a todas las respuestas de API:
    - timestamp: Momento de la respuesta
    - response_time_ms: Tiempo de procesamiento en milisegundos
    - api_version: Versión de la API
    """

    def __init__(self, app, api_version: str = "1.0.0"):
        super().__init__(app)
        self.api_version = api_version

    async def dispatch(self, request: Request, call_next):
        # Registrar tiempo de inicio
        start_time = time.time()

        # Procesar la petición
        response = await call_next(request)

        # Calcular tiempo de respuesta
        response_time_ms = int((time.time() - start_time) * 1000)

        # Solo procesar respuestas JSON de la API
        if request.url.path.startswith("/api/") and response.status_code < 400:

            # Para respuestas streaming, necesitamos leer el contenido
            if hasattr(response, "body_iterator"):
                try:
                    # Leer el contenido de la respuesta
                    body = b""
                    async for chunk in response.body_iterator:
                        body += chunk

                    # Intentar decodificar como JSON
                    try:
                        content = json.loads(body.decode())

                        # Si el contenido es un dict, añadir metadata
                        if isinstance(content, dict):
                            # Añadir metadata estándar
                            content["_metadata"] = {
                                "timestamp": utc_now().isoformat(),
                                "response_time_ms": response_time_ms,
                                "api_version": self.api_version,
                            }
                        else:
                            # Si es una lista u otro tipo, envolver en objeto con metadata
                            content = {
                                "data": content,
                                "_metadata": {
                                    "timestamp": utc_now().isoformat(),
                                    "response_time_ms": response_time_ms,
                                    "api_version": self.api_version,
                                },
                            }

                        # Crear nueva respuesta con el contenido modificado
                        return JSONResponse(
                            content=content,
                            status_code=response.status_code,
                            headers=dict(response.headers),
                        )
                    except json.JSONDecodeError:
                        # No es JSON, devolver tal cual
                        pass

                except Exception as e:
                    # Si hay algún error, devolver la respuesta original
                    logger.error(f"Error adding API standards: {e}")

        # Para respuestas no-JSON o errores, añadir headers
        response.headers["X-Response-Time-Ms"] = str(response_time_ms)
        response.headers["X-API-Version"] = self.api_version

        return response
