"""
Métricas Prometheus para sincronización CIMA
Issue #196 - Item #6
"""

import logging

logger = logging.getLogger(__name__)

# Prometheus client es opcional - no disponible en modo local (Windows desktop)
try:
    from prometheus_client import Counter, Gauge, Histogram
    PROMETHEUS_AVAILABLE = True
except ImportError:
    PROMETHEUS_AVAILABLE = False
    logger.info("[CIMA_METRICS] prometheus_client not available - metrics disabled (local mode)")


# ============================================================================
# STUB CLASSES PARA MODO LOCAL (cuando prometheus_client no está disponible)
# ============================================================================

class _StubMetric:
    """Stub métrica que no hace nada - para modo local sin prometheus"""
    def __init__(self, *args, **kwargs):
        pass

    def labels(self, *args, **kwargs):
        return self

    def inc(self, amount=1):
        pass

    def dec(self, amount=1):
        pass

    def set(self, value):
        pass

    def observe(self, value):
        pass


# ============================================================================
# MÉTRICAS (reales o stubs según disponibilidad)
# ============================================================================

if PROMETHEUS_AVAILABLE:
    # Duración de operaciones de sync CIMA
    cima_sync_duration_seconds = Histogram(
        "cima_sync_duration_seconds",
        "CIMA sync duration in seconds",
        ["operation"],  # Etiquetas: 'full_sync', 'streaming', 'chunked'
        buckets=(10, 30, 60, 120, 300, 600, 1800, 3600),  # 10s a 1h
    )

    # Total de items procesados
    cima_sync_items_total = Counter(
        "cima_sync_items_total",
        "Total number of CIMA items processed",
        ["operation", "status"],  # status: 'success', 'error'
    )

    # Total de errores en sync
    cima_sync_errors_total = Counter(
        "cima_sync_errors_total",
        "Total number of CIMA sync errors",
        ["operation", "error_type"],  # error_type: 'rate_limit', 'api_error', 'timeout', etc.
    )

    # Estado actual del sync (0=idle, 1=running)
    cima_sync_active = Gauge("cima_sync_active", "Whether CIMA sync is currently active (0=idle, 1=running)", ["operation"])

    # Progreso actual del sync (porcentaje)
    cima_sync_progress_percent = Gauge(
        "cima_sync_progress_percent", "Current progress of CIMA sync in percentage", ["operation"]
    )

    # ========== Métricas de Stall Detection (Issue #114) ==========

    # Total de stalls detectados
    cima_stalls_total = Counter(
        "cima_stalls_total",
        "Total number of CIMA stalls detected",
        ["page_range"],  # Rangos: '1-100', '101-200', '201-300', '301-400', '400+'
    )

    # Intentos de recovery de stall
    cima_recovery_attempts_total = Counter(
        "cima_recovery_attempts_total", "Total number of stall recovery attempts", ["success"]  # 'true' o 'false'
    )

    # Edad del último heartbeat (segundos desde última actualización)
    cima_heartbeat_age_seconds = Gauge("cima_heartbeat_age_seconds", "Seconds since last CIMA sync heartbeat")

    # Duración de operaciones de recovery
    cima_recovery_duration_seconds = Histogram(
        "cima_recovery_duration_seconds",
        "Duration of stall recovery operations in seconds",
        buckets=(1, 5, 10, 30, 60, 120, 300),  # 1s a 5min
    )

    # Estado del circuit breaker
    cima_circuit_breaker_state = Gauge(
        "cima_circuit_breaker_state", "Current state of CIMA circuit breaker (0=CLOSED, 1=OPEN, 2=HALF_OPEN)"
    )

    # Chunks consecutivos sin progreso (para debugging)
    cima_consecutive_same_page_chunks = Gauge(
        "cima_consecutive_same_page_chunks", "Number of consecutive chunks stuck on same page", ["page_number"]
    )

else:
    # Stubs - métricas que no hacen nada en modo local
    cima_sync_duration_seconds = _StubMetric()
    cima_sync_items_total = _StubMetric()
    cima_sync_errors_total = _StubMetric()
    cima_sync_active = _StubMetric()
    cima_sync_progress_percent = _StubMetric()
    cima_stalls_total = _StubMetric()
    cima_recovery_attempts_total = _StubMetric()
    cima_heartbeat_age_seconds = _StubMetric()
    cima_recovery_duration_seconds = _StubMetric()
    cima_circuit_breaker_state = _StubMetric()
    cima_consecutive_same_page_chunks = _StubMetric()
