# backend/app/measures/temporal_measures.py
"""
Medidas temporales farmacéuticas - Análisis en el tiempo estilo Power BI.

Medidas incluidas:
- MAT (Moving Annual Total): Últimos 12 meses móviles
- YTD (Year to Date): Desde enero hasta fecha actual
- MoM (Month over Month): Crecimiento mensual
- QoQ (Quarter over Quarter): Crecimiento trimestral
- QTD_YoY (Quarter-to-Date Year-over-Year): Variación trimestral interanual
- MTD_YoY (Month-to-Date Year-over-Year): Variación mensual interanual
- WTD_YoY (Week-to-Date Year-over-Year): Variación semanal interanual
"""

import calendar
from datetime import date, timedelta
from typing import Any, Dict

from dateutil.relativedelta import relativedelta


from .base import BaseMeasure, FilterContext, QueryContext
from .core_measures import TotalUnidades, TotalVentas


def _safe_date_previous_year(ref_date: date) -> date:
    """
    Create a date for the same day in the previous year, handling leap year edge case.

    If ref_date is Feb 29 and previous year is not a leap year, returns Feb 28.

    Args:
        ref_date: Reference date

    Returns:
        Same day/month in previous year (or adjusted for leap year)
    """
    try:
        return date(ref_date.year - 1, ref_date.month, ref_date.day)
    except ValueError:
        # Handle Feb 29 -> Feb 28 for non-leap years
        last_day = calendar.monthrange(ref_date.year - 1, ref_date.month)[1]
        return date(ref_date.year - 1, ref_date.month, min(ref_date.day, last_day))


def _calculate_yoy_variation(current: float, previous: float) -> float:
    """
    Calculate year-over-year percentage variation.

    Args:
        current: Current period value
        previous: Previous period value

    Returns:
        Percentage variation (100.0 if previous is 0 and current > 0)
    """
    if previous == 0:
        return 0.0 if current == 0 else 100.0
    return ((current - previous) / previous) * 100


class MAT(BaseMeasure):
    """
    Moving Annual Total - Últimos 12 meses móviles.
    Medida fundamental para análisis farmacéutico temporal.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ventas de los últimos 12 meses móviles"
        self.unit = "€"
        self.category = "Temporal"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """
        Calcular MAT desde la fecha de referencia.
        Si no hay fecha de fin, usa fecha actual.
        """
        # Determinar fecha de referencia
        end_date = context.filters.end_date or date.today()
        start_date = end_date - relativedelta(months=12)

        # Crear contexto temporal para MAT
        mat_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=start_date,
            end_date=end_date,
            # Mantener otros filtros
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        mat_context = QueryContext(context.db, mat_filters)

        # Calcular totales MAT
        total_ventas_mat = TotalVentas().calculate(mat_context)
        total_unidades_mat = TotalUnidades().calculate(mat_context)

        return {
            "periodo": {
                "inicio": start_date.isoformat(),
                "fin": end_date.isoformat(),
                "meses": 12,
            },
            "ventas_mat": total_ventas_mat,
            "unidades_mat": total_unidades_mat,
            "promedio_mensual_ventas": round(total_ventas_mat / 12, 2),
            "promedio_mensual_unidades": round(total_unidades_mat / 12, 2),
        }


class YTD(BaseMeasure):
    """
    Year to Date - Desde enero hasta fecha de referencia.
    Esencial para comparaciones anuales.
    """

    def __init__(self):
        super().__init__()
        self.description = "Ventas desde enero hasta la fecha"
        self.unit = "€"
        self.category = "Temporal"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular YTD desde enero del año actual/especificado"""
        # Determinar año de referencia
        end_date = context.filters.end_date or date.today()
        start_date = date(end_date.year, 1, 1)  # 1 enero del mismo año

        # Crear contexto YTD
        ytd_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=start_date,
            end_date=end_date,
            # Mantener otros filtros
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        ytd_context = QueryContext(context.db, ytd_filters)

        # Calcular totales YTD
        total_ventas_ytd = TotalVentas().calculate(ytd_context)
        total_unidades_ytd = TotalUnidades().calculate(ytd_context)

        # Calcular días transcurridos y promedio diario
        dias_transcurridos = (end_date - start_date).days + 1

        return {
            "periodo": {
                "inicio": start_date.isoformat(),
                "fin": end_date.isoformat(),
                "año": end_date.year,
                "dias_transcurridos": dias_transcurridos,
            },
            "ventas_ytd": total_ventas_ytd,
            "unidades_ytd": total_unidades_ytd,
            "promedio_diario_ventas": (
                round(total_ventas_ytd / dias_transcurridos, 2) if dias_transcurridos > 0 else 0
            ),
            "promedio_diario_unidades": (
                round(total_unidades_ytd / dias_transcurridos, 2) if dias_transcurridos > 0 else 0
            ),
        }


class MoM(BaseMeasure):
    """
    Month over Month - Crecimiento mensual.
    Compara mes actual vs mes anterior.
    """

    def __init__(self):
        super().__init__()
        self.description = "Crecimiento mensual (mes actual vs anterior)"
        self.unit = "%"
        self.category = "Crecimiento"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular crecimiento MoM"""
        # Determinar fecha de referencia
        end_date = context.filters.end_date or date.today()

        # Mes actual (o especificado)
        mes_actual_inicio = date(end_date.year, end_date.month, 1)
        mes_actual_fin = end_date

        # Mes anterior
        mes_anterior_fin = mes_actual_inicio - timedelta(days=1)
        mes_anterior_inicio = date(mes_anterior_fin.year, mes_anterior_fin.month, 1)

        # Contextos para ambos meses
        mes_actual_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=mes_actual_inicio,
            end_date=mes_actual_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        mes_anterior_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=mes_anterior_inicio,
            end_date=mes_anterior_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        mes_actual_context = QueryContext(context.db, mes_actual_filters)
        mes_anterior_context = QueryContext(context.db, mes_anterior_filters)

        # Calcular ventas de ambos meses
        ventas_mes_actual = TotalVentas().calculate(mes_actual_context)
        ventas_mes_anterior = TotalVentas().calculate(mes_anterior_context)

        # Calcular crecimiento
        if ventas_mes_anterior == 0:
            crecimiento_porcentual = 0.0 if ventas_mes_actual == 0 else 100.0
        else:
            crecimiento_porcentual = ((ventas_mes_actual - ventas_mes_anterior) / ventas_mes_anterior) * 100

        return {
            "mes_actual": {
                "periodo": f"{mes_actual_inicio.isoformat()} a {mes_actual_fin.isoformat()}",
                "ventas": ventas_mes_actual,
            },
            "mes_anterior": {
                "periodo": f"{mes_anterior_inicio.isoformat()} a {mes_anterior_fin.isoformat()}",
                "ventas": ventas_mes_anterior,
            },
            "crecimiento": {
                "absoluto": round(ventas_mes_actual - ventas_mes_anterior, 2),
                "porcentual": round(crecimiento_porcentual, 2),
                "tendencia": (
                    "positiva"
                    if crecimiento_porcentual > 0
                    else "negativa" if crecimiento_porcentual < 0 else "estable"
                ),
            },
        }


class QoQ(BaseMeasure):
    """
    Quarter over Quarter - Crecimiento trimestral.
    Compara trimestre actual vs trimestre anterior.
    """

    def __init__(self):
        super().__init__()
        self.description = "Crecimiento trimestral (trimestre actual vs anterior)"
        self.unit = "%"
        self.category = "Crecimiento"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular crecimiento QoQ"""
        end_date = context.filters.end_date or date.today()

        # Determinar trimestre actual
        trimestre_actual = ((end_date.month - 1) // 3) + 1

        # Fechas del trimestre actual
        if trimestre_actual == 1:  # Q1: Ene-Mar
            q_actual_inicio = date(end_date.year, 1, 1)
            q_actual_fin = date(end_date.year, 3, 31)
        elif trimestre_actual == 2:  # Q2: Abr-Jun
            q_actual_inicio = date(end_date.year, 4, 1)
            q_actual_fin = date(end_date.year, 6, 30)
        elif trimestre_actual == 3:  # Q3: Jul-Sep
            q_actual_inicio = date(end_date.year, 7, 1)
            q_actual_fin = date(end_date.year, 9, 30)
        else:  # Q4: Oct-Dic
            q_actual_inicio = date(end_date.year, 10, 1)
            q_actual_fin = date(end_date.year, 12, 31)

        # Ajustar fin de trimestre si la fecha de referencia es menor
        if end_date < q_actual_fin:
            q_actual_fin = end_date

        # Fechas del trimestre anterior
        if trimestre_actual == 1:  # Q1 -> Q4 año anterior
            q_anterior_inicio = date(end_date.year - 1, 10, 1)
            q_anterior_fin = date(end_date.year - 1, 12, 31)
        elif trimestre_actual == 2:  # Q2 -> Q1
            q_anterior_inicio = date(end_date.year, 1, 1)
            q_anterior_fin = date(end_date.year, 3, 31)
        elif trimestre_actual == 3:  # Q3 -> Q2
            q_anterior_inicio = date(end_date.year, 4, 1)
            q_anterior_fin = date(end_date.year, 6, 30)
        else:  # Q4 -> Q3
            q_anterior_inicio = date(end_date.year, 7, 1)
            q_anterior_fin = date(end_date.year, 9, 30)

        # Crear contextos para ambos trimestres
        q_actual_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=q_actual_inicio,
            end_date=q_actual_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        q_anterior_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=q_anterior_inicio,
            end_date=q_anterior_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        q_actual_context = QueryContext(context.db, q_actual_filters)
        q_anterior_context = QueryContext(context.db, q_anterior_filters)

        # Calcular ventas de ambos trimestres
        ventas_q_actual = TotalVentas().calculate(q_actual_context)
        ventas_q_anterior = TotalVentas().calculate(q_anterior_context)

        # Calcular crecimiento
        if ventas_q_anterior == 0:
            crecimiento_porcentual = 0.0 if ventas_q_actual == 0 else 100.0
        else:
            crecimiento_porcentual = ((ventas_q_actual - ventas_q_anterior) / ventas_q_anterior) * 100

        return {
            "trimestre_actual": {
                "trimestre": f"Q{trimestre_actual} {end_date.year}",
                "periodo": f"{q_actual_inicio.isoformat()} a {q_actual_fin.isoformat()}",
                "ventas": ventas_q_actual,
            },
            "trimestre_anterior": {
                "trimestre": f"Q{trimestre_actual-1 if trimestre_actual > 1 else 4} {end_date.year if trimestre_actual > 1 else end_date.year-1}",
                "periodo": f"{q_anterior_inicio.isoformat()} a {q_anterior_fin.isoformat()}",
                "ventas": ventas_q_anterior,
            },
            "crecimiento": {
                "absoluto": round(ventas_q_actual - ventas_q_anterior, 2),
                "porcentual": round(crecimiento_porcentual, 2),
                "tendencia": (
                    "positiva"
                    if crecimiento_porcentual > 0
                    else "negativa" if crecimiento_porcentual < 0 else "estable"
                ),
            },
        }


class QTD_YoY(BaseMeasure):
    """
    Quarter-to-Date Year-over-Year.
    Compara período del trimestre actual vs mismo período del año anterior.
    """

    def __init__(self):
        super().__init__()
        self.description = "Variación del trimestre actual vs mismo período año anterior"
        self.unit = "%"
        self.category = "YoY"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular QTD YoY - Quarter-to-Date vs mismo período año anterior"""
        end_date = context.filters.end_date or date.today()

        # Determinar trimestre actual
        trimestre = ((end_date.month - 1) // 3) + 1

        # Fechas del trimestre actual (desde inicio del trimestre hasta fecha actual)
        if trimestre == 1:
            q_inicio = date(end_date.year, 1, 1)
        elif trimestre == 2:
            q_inicio = date(end_date.year, 4, 1)
        elif trimestre == 3:
            q_inicio = date(end_date.year, 7, 1)
        else:
            q_inicio = date(end_date.year, 10, 1)

        # Período actual (QTD)
        actual_inicio = q_inicio
        actual_fin = end_date

        # Mismo período del año anterior (usando helper para leap year)
        anterior_inicio = _safe_date_previous_year(q_inicio)
        anterior_fin = _safe_date_previous_year(end_date)

        # Crear contextos para ambos períodos
        actual_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=actual_inicio,
            end_date=actual_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        anterior_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=anterior_inicio,
            end_date=anterior_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        actual_context = QueryContext(context.db, actual_filters)
        anterior_context = QueryContext(context.db, anterior_filters)

        # Calcular ventas y unidades de ambos períodos
        ventas_actual = TotalVentas().calculate(actual_context)
        ventas_anterior = TotalVentas().calculate(anterior_context)
        unidades_actual = TotalUnidades().calculate(actual_context)
        unidades_anterior = TotalUnidades().calculate(anterior_context)

        # Calcular variaciones (usando helper DRY)
        variacion_ventas = _calculate_yoy_variation(ventas_actual, ventas_anterior)
        variacion_unidades = _calculate_yoy_variation(unidades_actual, unidades_anterior)

        return {
            "tipo": "QTD",
            "periodo_actual": {
                "trimestre": f"Q{trimestre} {end_date.year}",
                "inicio": actual_inicio.isoformat(),
                "fin": actual_fin.isoformat(),
                "ventas": ventas_actual,
                "unidades": unidades_actual,
            },
            "periodo_anterior": {
                "trimestre": f"Q{trimestre} {end_date.year - 1}",
                "inicio": anterior_inicio.isoformat(),
                "fin": anterior_fin.isoformat(),
                "ventas": ventas_anterior,
                "unidades": unidades_anterior,
            },
            "variacion": {
                "ventas_porcentual": round(variacion_ventas, 2),
                "ventas_absoluta": round(ventas_actual - ventas_anterior, 2),
                "unidades_porcentual": round(variacion_unidades, 2),
                "unidades_absoluta": round(unidades_actual - unidades_anterior, 2),
                "tendencia": (
                    "positiva"
                    if variacion_ventas > 0
                    else "negativa" if variacion_ventas < 0 else "estable"
                ),
            },
        }


class MTD_YoY(BaseMeasure):
    """
    Month-to-Date Year-over-Year.
    Compara período del mes actual vs mismo período del año anterior.
    """

    def __init__(self):
        super().__init__()
        self.description = "Variación del mes actual vs mismo período año anterior"
        self.unit = "%"
        self.category = "YoY"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular MTD YoY - Month-to-Date vs mismo período año anterior"""
        end_date = context.filters.end_date or date.today()

        # Período actual (desde inicio del mes hasta fecha actual)
        actual_inicio = date(end_date.year, end_date.month, 1)
        actual_fin = end_date

        # Mismo período del año anterior (usando helper para leap year)
        anterior_inicio = date(end_date.year - 1, end_date.month, 1)
        anterior_fin = _safe_date_previous_year(end_date)

        # Crear contextos para ambos períodos
        actual_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=actual_inicio,
            end_date=actual_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        anterior_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=anterior_inicio,
            end_date=anterior_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        actual_context = QueryContext(context.db, actual_filters)
        anterior_context = QueryContext(context.db, anterior_filters)

        # Calcular ventas y unidades de ambos períodos
        ventas_actual = TotalVentas().calculate(actual_context)
        ventas_anterior = TotalVentas().calculate(anterior_context)
        unidades_actual = TotalUnidades().calculate(actual_context)
        unidades_anterior = TotalUnidades().calculate(anterior_context)

        # Calcular variaciones (usando helper DRY)
        variacion_ventas = _calculate_yoy_variation(ventas_actual, ventas_anterior)
        variacion_unidades = _calculate_yoy_variation(unidades_actual, unidades_anterior)

        # Nombre del mes en español
        meses = ["Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio",
                 "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"]
        mes_nombre = meses[end_date.month - 1]

        return {
            "tipo": "MTD",
            "periodo_actual": {
                "mes": f"{mes_nombre} {end_date.year}",
                "inicio": actual_inicio.isoformat(),
                "fin": actual_fin.isoformat(),
                "ventas": ventas_actual,
                "unidades": unidades_actual,
            },
            "periodo_anterior": {
                "mes": f"{mes_nombre} {end_date.year - 1}",
                "inicio": anterior_inicio.isoformat(),
                "fin": anterior_fin.isoformat(),
                "ventas": ventas_anterior,
                "unidades": unidades_anterior,
            },
            "variacion": {
                "ventas_porcentual": round(variacion_ventas, 2),
                "ventas_absoluta": round(ventas_actual - ventas_anterior, 2),
                "unidades_porcentual": round(variacion_unidades, 2),
                "unidades_absoluta": round(unidades_actual - unidades_anterior, 2),
                "tendencia": (
                    "positiva"
                    if variacion_ventas > 0
                    else "negativa" if variacion_ventas < 0 else "estable"
                ),
            },
        }


class WTD_YoY(BaseMeasure):
    """
    Week-to-Date Year-over-Year.
    Compara período de la semana actual vs misma semana del año anterior.
    """

    def __init__(self):
        super().__init__()
        self.description = "Variación de la semana actual vs misma semana año anterior"
        self.unit = "%"
        self.category = "YoY"

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular WTD YoY - Week-to-Date vs misma semana año anterior"""
        end_date = context.filters.end_date or date.today()

        # Período actual (desde inicio de la semana -lunes- hasta fecha actual)
        dia_semana = end_date.weekday()  # 0 = lunes
        actual_inicio = end_date - timedelta(days=dia_semana)
        actual_fin = end_date

        # Misma semana del año anterior (ISO week)
        iso_year, iso_week, _ = end_date.isocalendar()
        # Encontrar el lunes de la misma semana ISO del año anterior
        primera_semana_anterior = date(iso_year - 1, 1, 4)  # 4 enero siempre está en semana 1
        primer_lunes_anterior = primera_semana_anterior - timedelta(days=primera_semana_anterior.weekday())
        anterior_inicio = primer_lunes_anterior + timedelta(weeks=iso_week - 1)
        anterior_fin = anterior_inicio + timedelta(days=dia_semana)

        # Crear contextos para ambos períodos
        actual_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=actual_inicio,
            end_date=actual_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        anterior_filters = FilterContext(
            pharmacy_id=context.filters.pharmacy_id,
            start_date=anterior_inicio,
            end_date=anterior_fin,
            product_codes=context.filters.product_codes,
            therapeutic_categories=context.filters.therapeutic_categories,
            laboratories=context.filters.laboratories,
            requires_prescription=context.filters.requires_prescription,
            is_generic=context.filters.is_generic,
        )

        actual_context = QueryContext(context.db, actual_filters)
        anterior_context = QueryContext(context.db, anterior_filters)

        # Calcular ventas y unidades de ambos períodos
        ventas_actual = TotalVentas().calculate(actual_context)
        ventas_anterior = TotalVentas().calculate(anterior_context)
        unidades_actual = TotalUnidades().calculate(actual_context)
        unidades_anterior = TotalUnidades().calculate(anterior_context)

        # Calcular variaciones (usando helper DRY)
        variacion_ventas = _calculate_yoy_variation(ventas_actual, ventas_anterior)
        variacion_unidades = _calculate_yoy_variation(unidades_actual, unidades_anterior)

        return {
            "tipo": "WTD",
            "periodo_actual": {
                "semana": f"Semana {iso_week} {end_date.year}",
                "inicio": actual_inicio.isoformat(),
                "fin": actual_fin.isoformat(),
                "ventas": ventas_actual,
                "unidades": unidades_actual,
            },
            "periodo_anterior": {
                "semana": f"Semana {iso_week} {end_date.year - 1}",
                "inicio": anterior_inicio.isoformat(),
                "fin": anterior_fin.isoformat(),
                "ventas": ventas_anterior,
                "unidades": unidades_anterior,
            },
            "variacion": {
                "ventas_porcentual": round(variacion_ventas, 2),
                "ventas_absoluta": round(ventas_actual - ventas_anterior, 2),
                "unidades_porcentual": round(variacion_unidades, 2),
                "unidades_absoluta": round(unidades_actual - unidades_anterior, 2),
                "tendencia": (
                    "positiva"
                    if variacion_ventas > 0
                    else "negativa" if variacion_ventas < 0 else "estable"
                ),
            },
        }
