# backend/app/measures/rotation_measures.py
"""
Medidas de rotación de inventario - Issue #470.

Medidas incluidas:
- RotationIndex: Índice de rotación anual
- DaysInventory: Días promedio en inventario
- ABCClassification: Clasificación ABC por ventas/rotación
"""

from datetime import date, timedelta
from typing import Any, Dict, List, Union

from sqlalchemy import func, desc

from app.models.inventory_snapshot import InventorySnapshot
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseMeasure, QueryContext


# Issue #500: Mapeo de tipos de producto (frontend → DB)
PRODUCT_TYPE_MAP = {
    "medicamento": "prescription",
    "prescription": "prescription",
    "venta_libre": "venta_libre",
    "otc": "venta_libre",
}


class RotationIndex(BaseMeasure):
    """
    Índice de rotación anual.

    Fórmula: Coste de ventas 12 meses / Stock promedio a coste
    Interpretación: >4 = buena rotación, <2 = rotación lenta
    """

    def __init__(self):
        super().__init__()
        self.description = "Índice de rotación anual del inventario"
        self.unit = "veces/año"
        self.category = "Rotación"
        self.months_back = 12  # Período de análisis

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular índice de rotación."""
        end_date = date.today()
        start_date = end_date - timedelta(days=365)

        # 1. Calcular coste de ventas (12 meses)
        # Primero intentar con purchase_price, si no hay datos usar total_amount con margen estimado
        cogs_query = context.db.query(
            func.sum(SalesData.purchase_price * SalesData.quantity).label("cogs"),
            func.sum(SalesData.total_amount).label("total_sales"),
        ).filter(
            SalesData.pharmacy_id == context.filters.pharmacy_id,
            SalesData.sale_date >= start_date,
            SalesData.sale_date <= end_date,
        )

        # Issue #500: Filtrar por product_type (requiere join con SalesEnrichment)
        if context.filters.product_type:
            db_type = PRODUCT_TYPE_MAP.get(context.filters.product_type, context.filters.product_type)
            cogs_query = cogs_query.join(
                SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
            ).filter(SalesEnrichment.product_type == db_type)

        # Apply product_codes filter if specified (Issue #470 code review fix)
        if context.filters.product_codes:
            cogs_query = cogs_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        cogs_result = cogs_query.one()

        # Si no hay purchase_price, estimar COGS con margen típico farmacia (30%)
        if cogs_result.cogs and float(cogs_result.cogs) > 0:
            cogs = float(cogs_result.cogs)
        elif cogs_result.total_sales:
            # Margen típico farmacia: 30% → COGS = ventas * 0.70
            cogs = float(cogs_result.total_sales) * 0.70
        else:
            cogs = 0

        # 2. Obtener stock actual a coste
        stock_result = context.inventory_query.with_entities(
            func.sum(
                InventorySnapshot.stock_quantity *
                func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
            ).label("stock_value")
        ).scalar()

        avg_stock = float(stock_result or 0)

        if avg_stock == 0:
            return {
                "rotation_index": 0,
                "cogs": cogs,
                "avg_stock": 0,
                "interpretation": "Sin datos de inventario"
            }

        rotation_index = cogs / avg_stock

        # Interpretación según estándares farmacéuticos
        if rotation_index >= 6:
            interpretation = "Excelente rotación"
        elif rotation_index >= 4:
            interpretation = "Buena rotación"
        elif rotation_index >= 2:
            interpretation = "Rotación moderada"
        elif rotation_index >= 1:
            interpretation = "Rotación lenta"
        else:
            interpretation = "Rotación muy lenta - Revisar surtido"

        return {
            "rotation_index": round(rotation_index, 2),
            "cogs": round(cogs, 2),
            "avg_stock": round(avg_stock, 2),
            "period_days": 365,
            "interpretation": interpretation,
        }


class DaysInventory(BaseMeasure):
    """
    Días promedio en inventario (DIO - Days Inventory Outstanding).

    Fórmula: 365 / Rotation Index
    Interpretación: <90 días = saludable, >180 días = problema
    """

    def __init__(self):
        super().__init__()
        self.description = "Días promedio que el inventario permanece en stock"
        self.unit = "días"
        self.category = "Rotación"
        self.dependencies = ["RotationIndex"]

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular días promedio en inventario."""
        # Usar RotationIndex para el cálculo
        rotation_result = RotationIndex().calculate(context)

        if isinstance(rotation_result, dict):
            rotation_index = rotation_result.get("rotation_index", 0)
        else:
            rotation_index = rotation_result

        if rotation_index == 0:
            return {
                "days_inventory": None,  # JSON-serializable (not float("inf"))
                "rotation_index": 0,
                "interpretation": "Sin rotación calculable - Días indefinidos"
            }

        days_inventory = 365 / rotation_index

        # Interpretación
        if days_inventory <= 60:
            interpretation = "Rotación excelente"
        elif days_inventory <= 90:
            interpretation = "Rotación saludable"
        elif days_inventory <= 120:
            interpretation = "Rotación aceptable"
        elif days_inventory <= 180:
            interpretation = "Rotación lenta - Revisar"
        else:
            interpretation = "Inventario estancado"

        return {
            "days_inventory": round(days_inventory, 1),
            "rotation_index": round(rotation_index, 2),
            "interpretation": interpretation,
        }


class ABCClassification(BaseMeasure):
    """
    Clasificación ABC por ventas (Pareto).

    Clasificación:
    - A: Top 80% de ventas (~20% productos)
    - B: Siguiente 15% de ventas (~30% productos)
    - C: Último 5% de ventas (~50% productos)
    """

    def __init__(self):
        super().__init__()
        self.description = "Clasificación ABC de productos por ventas"
        self.unit = "clasificación"
        self.category = "Rotación"
        # Umbrales Pareto estándar
        self.class_a_threshold = 0.80  # 80% de ventas
        self.class_b_threshold = 0.95  # 80% + 15% = 95%

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular clasificación ABC de productos."""
        end_date = date.today()
        start_date = end_date - timedelta(days=365)

        # Obtener ventas por producto (últimos 12 meses)
        sales_query = context.db.query(
            SalesData.codigo_nacional,
            func.max(SalesData.product_name).label("product_name"),  # Take any name (avoid duplicates)
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.quantity).label("total_units"),
        ).filter(
            SalesData.pharmacy_id == context.filters.pharmacy_id,
            SalesData.sale_date >= start_date,
            SalesData.sale_date <= end_date,
            SalesData.codigo_nacional.isnot(None),  # Exclude products without CN
        )

        # Issue #500: Filtrar por product_type (requiere join con SalesEnrichment)
        if context.filters.product_type:
            db_type = PRODUCT_TYPE_MAP.get(context.filters.product_type, context.filters.product_type)
            sales_query = sales_query.join(
                SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
            ).filter(SalesEnrichment.product_type == db_type)

        # Apply product_codes filter if specified (Issue #470 code review fix)
        if context.filters.product_codes:
            sales_query = sales_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        sales_by_product = sales_query.group_by(
            SalesData.codigo_nacional,  # Only group by CN to avoid duplicates
        ).order_by(
            desc("total_sales")
        ).all()

        if not sales_by_product:
            return {
                "class_a": {"products": 0, "sales_pct": 0, "products_pct": 0},
                "class_b": {"products": 0, "sales_pct": 0, "products_pct": 0},
                "class_c": {"products": 0, "sales_pct": 0, "products_pct": 0},
                "total_products": 0,
                "total_sales": 0,
            }

        # Calcular totales (convertir a float para evitar errores Decimal/float)
        total_sales = float(sum(p.total_sales or 0 for p in sales_by_product))
        total_products = len(sales_by_product)

        # Obtener stock por producto para calcular rotación individual
        # Usar el snapshot más reciente
        latest_date_subq = (
            context.db.query(func.max(InventorySnapshot.snapshot_date))
            .filter(InventorySnapshot.pharmacy_id == context.filters.pharmacy_id)
            .scalar_subquery()
        )

        # Issue #500: Filtrar inventario por product_type también
        inventory_query = context.db.query(
            InventorySnapshot.product_code,
            InventorySnapshot.stock_quantity,
            (InventorySnapshot.stock_quantity *
             func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)).label("stock_value")
        ).filter(
            InventorySnapshot.pharmacy_id == context.filters.pharmacy_id,
            InventorySnapshot.snapshot_date == latest_date_subq,
        )

        if context.filters.product_type:
            db_type = PRODUCT_TYPE_MAP.get(context.filters.product_type, context.filters.product_type)
            inventory_query = inventory_query.filter(InventorySnapshot.product_type == db_type)

        inventory_by_product = {
            row.product_code: {
                "stock_qty": int(row.stock_quantity or 0),
                "stock_value": float(row.stock_value or 0),
            }
            for row in inventory_query.all()
        }

        # Clasificar productos
        class_a: List[Dict] = []
        class_b: List[Dict] = []
        class_c: List[Dict] = []

        cumulative_sales = 0
        cumulative_pct = 0

        for product in sales_by_product:
            product_sales = float(product.total_sales or 0)
            cumulative_sales += product_sales
            cumulative_pct = cumulative_sales / total_sales if total_sales > 0 else 0

            # Obtener datos de inventario para este producto
            inv_data = inventory_by_product.get(product.codigo_nacional, {})
            stock_value = inv_data.get("stock_value", 0)
            stock_qty = inv_data.get("stock_qty", 0)

            # Calcular rotación por producto (ventas/stock a coste)
            # Si no hay stock, usar COGS estimado (ventas * 0.70)
            product_cogs = product_sales * 0.70  # Margen típico 30%
            if stock_value > 0:
                rotation = round(product_cogs / stock_value, 2)
                days_inventory = round(365 / rotation, 0) if rotation > 0 else None
            else:
                rotation = None
                days_inventory = None

            # Calcular GMROI por producto
            gross_margin = product_sales * 0.30  # Margen estimado 30%
            gmroi = round(gross_margin / stock_value, 2) if stock_value > 0 else None

            # Determinar clase ABC
            if cumulative_pct <= self.class_a_threshold:
                abc_class = "A"
            elif cumulative_pct <= self.class_b_threshold:
                abc_class = "B"
            else:
                abc_class = "C"

            product_data = {
                "codigo_nacional": product.codigo_nacional,
                "product_name": product.product_name,
                "sales": product_sales,
                "units": int(product.total_units or 0),
                "stock_qty": stock_qty,
                "rotation": rotation,
                "days_inventory": days_inventory,
                "gmroi": gmroi,
                "abc_class": abc_class,
            }

            if abc_class == "A":
                class_a.append(product_data)
            elif abc_class == "B":
                class_b.append(product_data)
            else:
                class_c.append(product_data)

        # Calcular estadísticas por clase
        def class_stats(products: List[Dict], total_sales: float, total_products: int):
            class_sales = sum(p["sales"] for p in products)
            return {
                "products": len(products),
                "products_pct": round(len(products) / total_products * 100, 1) if total_products > 0 else 0,
                "sales": round(class_sales, 2),
                "sales_pct": round(class_sales / total_sales * 100, 1) if total_sales > 0 else 0,
            }

        # Issue #533: Filtrar por clase ABC si se especificó en contexto
        # Esto permite obtener productos B/C sin que sean excluidos por el límite
        if context.filters.abc_class:
            abc_filter_value = context.filters.abc_class.upper()
            if abc_filter_value == "A":
                filtered_products = class_a
            elif abc_filter_value == "B":
                filtered_products = class_b
            elif abc_filter_value == "C":
                filtered_products = class_c
            else:
                # "all" o valor inválido: incluir todos
                filtered_products = class_a + class_b + class_c
        else:
            # Sin filtro: combinar todas las clases
            filtered_products = class_a + class_b + class_c

        # Limitar a 100 productos DESPUÉS de filtrar por clase
        all_products_limited = filtered_products[:100]

        return {
            "class_a": class_stats(class_a, total_sales, total_products),
            "class_b": class_stats(class_b, total_sales, total_products),
            "class_c": class_stats(class_c, total_sales, total_products),
            "total_products": total_products,
            "total_sales": round(total_sales, 2),
            "thresholds": {
                "a": self.class_a_threshold,
                "b": self.class_b_threshold,
            },
            # Top 10 productos clase A (legacy, mantener compatibilidad)
            "top_a_products": all_products_limited,
        }
