# backend/app/measures/profitability_measures.py
"""
Medidas de rentabilidad farmacéutica - Issue #470.

Medidas incluidas:
- ROIProduct: ROI por producto
- GMROI: Gross Margin Return on Investment
- ContributionMargin: Margen de contribución
"""

from datetime import date, timedelta
from typing import Any, Dict, Union

from sqlalchemy import func

from app.models.inventory_snapshot import InventorySnapshot
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment

from .base import BaseMeasure, QueryContext


# Issue #500: Mapeo de tipos de producto (frontend → DB)
PRODUCT_TYPE_MAP = {
    "medicamento": "prescription",
    "prescription": "prescription",
    "venta_libre": "venta_libre",
    "otc": "venta_libre",
}


class ROIProduct(BaseMeasure):
    """
    ROI por producto - Retorno sobre inversión en inventario.

    Fórmula: (Margen bruto anual / Inversión en stock) × 100
    Interpretación: >100% = inversión recuperada en el año
    """

    def __init__(self):
        super().__init__()
        self.description = "Retorno sobre inversión en inventario (%)"
        self.unit = "%"
        self.category = "Rentabilidad"

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular ROI del inventario."""
        end_date = date.today()
        start_date = end_date - timedelta(days=365)

        # 1. Calcular margen bruto anual
        # Margen = Ventas - Coste de ventas
        sales_query = context.db.query(
            func.sum(SalesData.total_amount).label("total_sales"),
            func.sum(SalesData.purchase_price * SalesData.quantity).label("cogs"),
        ).filter(
            SalesData.pharmacy_id == context.filters.pharmacy_id,
            SalesData.sale_date >= start_date,
            SalesData.sale_date <= end_date,
        )

        # Issue #500: Filtrar por product_type (requiere join con SalesEnrichment)
        if context.filters.product_type:
            db_type = PRODUCT_TYPE_MAP.get(context.filters.product_type, context.filters.product_type)
            sales_query = sales_query.join(
                SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
            ).filter(SalesEnrichment.product_type == db_type)

        # Apply product_codes filter if specified (Issue #470 code review fix)
        if context.filters.product_codes:
            sales_query = sales_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        sales_result = sales_query.first()

        total_sales = float(sales_result.total_sales or 0) if sales_result else 0
        cogs = float(sales_result.cogs or 0) if sales_result else 0
        gross_margin = total_sales - cogs

        # 2. Obtener inversión en stock (valor del inventario a coste)
        stock_result = context.inventory_query.with_entities(
            func.sum(
                InventorySnapshot.stock_quantity *
                func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
            ).label("stock_investment")
        ).scalar()

        stock_investment = float(stock_result or 0)

        if stock_investment == 0:
            return {
                "roi_percent": 0,
                "gross_margin": gross_margin,
                "stock_investment": 0,
                "interpretation": "Sin datos de inversión"
            }

        roi = (gross_margin / stock_investment) * 100

        # Interpretación
        if roi >= 200:
            interpretation = "ROI excelente"
        elif roi >= 100:
            interpretation = "ROI bueno - Inversión recuperada"
        elif roi >= 50:
            interpretation = "ROI moderado"
        elif roi >= 0:
            interpretation = "ROI bajo - Revisar rentabilidad"
        else:
            interpretation = "ROI negativo - Pérdidas"

        return {
            "roi_percent": round(roi, 2),
            "gross_margin": round(gross_margin, 2),
            "stock_investment": round(stock_investment, 2),
            "total_sales": round(total_sales, 2),
            "cogs": round(cogs, 2),
            "period_days": 365,
            "interpretation": interpretation,
        }


class GMROI(BaseMeasure):
    """
    Gross Margin Return on Investment.

    Fórmula: Margen bruto / Inventario promedio a coste
    Interpretación: >1.5 = rentable, <1.0 = problema
    Estándar retail farmacéutico: 2.0 - 3.0

    Issue #496: Usa 3-tier fallback para COGS:
    1. InventorySnapshot.unit_cost (PMC) - Coste real del inventario
    2. SalesData.purchase_price - Si no hay match de inventario
    3. Estimación 30% margen - Último recurso
    """

    def __init__(self):
        super().__init__()
        self.description = "Retorno del margen bruto sobre inversión en inventario"
        self.unit = "ratio"
        self.category = "Rentabilidad"

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """
        Calcular GMROI usando costes reales del inventario (PMC).

        Issue #496: 3-tier fallback para COGS más preciso:
        1. InventorySnapshot.unit_cost (PMC) - Prioridad máxima
        2. SalesData.purchase_price - Si no hay match de inventario
        3. Estimación (ventas * 0.70) - Último recurso
        """
        from sqlalchemy import case, and_, or_
        from sqlalchemy.sql import literal_column

        end_date = date.today()
        start_date = end_date - timedelta(days=365)

        # Obtener subquery de costes de inventario (PMC del snapshot más reciente)
        inv_costs = context.get_latest_inventory_costs_subquery()

        # Query con LEFT JOIN a inventory costs para 3-tier fallback
        # COGS = SUM(CASE WHEN inventory_cost > 0 THEN inv_cost * qty
        #            WHEN purchase_price > 0 THEN purchase_price * qty
        #            ELSE total_amount * 0.70 END)
        sales_query = (
            context.db.query(
                func.sum(SalesData.total_amount).label("total_sales"),
                func.sum(SalesData.quantity).label("total_units"),
                # 3-tier fallback COGS calculation
                func.sum(
                    case(
                        # Tier 1: Inventory cost (PMC) - prioridad máxima
                        (
                            and_(
                                inv_costs.c.inventory_unit_cost.isnot(None),
                                inv_costs.c.inventory_unit_cost > 0
                            ),
                            inv_costs.c.inventory_unit_cost * SalesData.quantity
                        ),
                        # Tier 2: Purchase price from sales data
                        (
                            and_(
                                SalesData.purchase_price.isnot(None),
                                SalesData.purchase_price > 0
                            ),
                            SalesData.purchase_price * SalesData.quantity
                        ),
                        # Tier 3: Estimation (70% of sale amount = 30% margin)
                        else_=SalesData.total_amount * 0.70
                    )
                ).label("cogs"),
                # Data quality metrics - units con cada tier
                func.sum(
                    case(
                        (
                            and_(
                                inv_costs.c.inventory_unit_cost.isnot(None),
                                inv_costs.c.inventory_unit_cost > 0
                            ),
                            SalesData.quantity
                        ),
                        else_=literal_column("0")
                    )
                ).label("units_with_inventory_cost"),
                func.sum(
                    case(
                        (
                            and_(
                                or_(
                                    inv_costs.c.inventory_unit_cost.is_(None),
                                    inv_costs.c.inventory_unit_cost == 0
                                ),
                                SalesData.purchase_price.isnot(None),
                                SalesData.purchase_price > 0
                            ),
                            SalesData.quantity
                        ),
                        else_=literal_column("0")
                    )
                ).label("units_with_purchase_price"),
            )
            .outerjoin(
                inv_costs,
                or_(
                    SalesData.codigo_nacional == inv_costs.c.product_code,
                    and_(
                        SalesData.ean13.isnot(None),
                        SalesData.ean13 == inv_costs.c.ean13
                    )
                )
            )
            .filter(
                SalesData.pharmacy_id == context.filters.pharmacy_id,
                SalesData.sale_date >= start_date,
                SalesData.sale_date <= end_date,
            )
        )

        # Issue #500: Filtrar por product_type (requiere join con SalesEnrichment)
        if context.filters.product_type:
            db_type = PRODUCT_TYPE_MAP.get(context.filters.product_type, context.filters.product_type)
            sales_query = sales_query.join(
                SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id
            ).filter(SalesEnrichment.product_type == db_type)

        # Apply product_codes filter if specified
        if context.filters.product_codes:
            sales_query = sales_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        sales_result = sales_query.first()

        total_sales = float(sales_result.total_sales or 0) if sales_result else 0
        total_units = int(sales_result.total_units or 0) if sales_result else 0
        cogs = float(sales_result.cogs or 0) if sales_result else 0
        units_with_inv = int(sales_result.units_with_inventory_cost or 0) if sales_result else 0
        units_with_pp = int(sales_result.units_with_purchase_price or 0) if sales_result else 0

        # Calculate data quality metrics
        inv_cost_coverage = round((units_with_inv / total_units * 100), 1) if total_units > 0 else 0
        pp_fallback_pct = round((units_with_pp / total_units * 100), 1) if total_units > 0 else 0
        estimated_pct = round(100 - inv_cost_coverage - pp_fallback_pct, 1)

        gross_margin = total_sales - cogs

        # 2. Obtener inventario promedio a coste
        stock_result = context.inventory_query.with_entities(
            func.sum(
                InventorySnapshot.stock_quantity *
                func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
            ).label("avg_inventory")
        ).scalar()

        avg_inventory = float(stock_result or 0)

        if avg_inventory == 0:
            return {
                "gmroi": 0,
                "gross_margin": gross_margin,
                "avg_inventory": 0,
                "interpretation": "Sin datos de inventario",
                "data_quality": {
                    "cost_source": "no_inventory",
                    "inventory_cost_coverage_pct": 0,
                }
            }

        gmroi = gross_margin / avg_inventory

        # Interpretación según estándares farmacéuticos
        if gmroi >= 3.0:
            interpretation = "GMROI excelente"
        elif gmroi >= 2.0:
            interpretation = "GMROI muy bueno"
        elif gmroi >= 1.5:
            interpretation = "GMROI saludable"
        elif gmroi >= 1.0:
            interpretation = "GMROI bajo - Optimizar"
        else:
            interpretation = "GMROI crítico - Pérdidas"

        # Calcular margen porcentual para contexto
        margin_pct = (gross_margin / total_sales * 100) if total_sales > 0 else 0

        return {
            "gmroi": round(gmroi, 2),
            "gross_margin": round(gross_margin, 2),
            "avg_inventory": round(avg_inventory, 2),
            "total_sales": round(total_sales, 2),
            "cogs": round(cogs, 2),
            "margin_percent": round(margin_pct, 2),
            "interpretation": interpretation,
            "benchmark": "Farmacia óptimo: 2.0 - 3.0",
            # Issue #496: Data quality metrics
            "data_quality": {
                "total_units": total_units,
                "inventory_cost_coverage_pct": inv_cost_coverage,
                "purchase_price_fallback_pct": pp_fallback_pct,
                "estimated_pct": estimated_pct,
                "cost_source_priority": "InventorySnapshot.unit_cost > SalesData.purchase_price > estimated",
            },
        }


class ContributionMargin(BaseMeasure):
    """
    Margen de contribución.

    Fórmula: (PVP - Coste variable) / PVP × 100
    También conocido como: Margen bruto porcentual
    Interpretación: >25% = saludable para farmacia
    """

    def __init__(self):
        super().__init__()
        self.description = "Margen de contribución porcentual"
        self.unit = "%"
        self.category = "Rentabilidad"

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular margen de contribución."""
        end_date = date.today()
        start_date = end_date - timedelta(days=365)

        # Calcular totales de ventas y costes
        sales_query = context.db.query(
            func.sum(SalesData.total_amount).label("total_revenue"),
            func.sum(SalesData.purchase_price * SalesData.quantity).label("variable_costs"),
            func.count(SalesData.id).label("num_transactions"),
        ).filter(
            SalesData.pharmacy_id == context.filters.pharmacy_id,
            SalesData.sale_date >= start_date,
            SalesData.sale_date <= end_date,
        )

        # Apply product_codes filter if specified (Issue #470 code review fix)
        if context.filters.product_codes:
            sales_query = sales_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        sales_result = sales_query.first()

        total_revenue = float(sales_result.total_revenue or 0) if sales_result else 0
        variable_costs = float(sales_result.variable_costs or 0) if sales_result else 0
        num_transactions = int(sales_result.num_transactions or 0) if sales_result else 0

        if total_revenue == 0:
            return {
                "contribution_margin_pct": 0,
                "contribution_margin_abs": 0,
                "total_revenue": 0,
                "variable_costs": 0,
                "interpretation": "Sin datos de ventas"
            }

        contribution_margin = total_revenue - variable_costs
        contribution_margin_pct = (contribution_margin / total_revenue) * 100

        # Interpretación para farmacia
        if contribution_margin_pct >= 35:
            interpretation = "Margen excelente"
        elif contribution_margin_pct >= 28:
            interpretation = "Margen muy bueno"
        elif contribution_margin_pct >= 22:
            interpretation = "Margen saludable"
        elif contribution_margin_pct >= 15:
            interpretation = "Margen ajustado"
        else:
            interpretation = "Margen crítico - Revisar precios/proveedores"

        return {
            "contribution_margin_pct": round(contribution_margin_pct, 2),
            "contribution_margin_abs": round(contribution_margin, 2),
            "total_revenue": round(total_revenue, 2),
            "variable_costs": round(variable_costs, 2),
            "num_transactions": num_transactions,
            "period_days": 365,
            "interpretation": interpretation,
            "benchmark": "Farmacia óptimo: 25-30%",
        }
