# backend/app/measures/inventory_measures.py
"""
Medidas de inventario farmacéutico - Issue #470.

Medidas incluidas:
- StockValue: Valor total del inventario a coste
- DeadStockValue: Valor del stock sin movimiento > 180 días
- StockCoverageDays: Días de cobertura del stock actual
- StockAlerts: Lista de productos con problemas de stock
"""

from datetime import date, timedelta
from typing import Any, Dict, List, Union

from sqlalchemy import func, case, desc, or_

from app.models.inventory_snapshot import InventorySnapshot
from app.models.sales_data import SalesData

from .base import BaseMeasure, QueryContext


class StockValue(BaseMeasure):
    """
    Valor total del inventario a coste.

    Fórmula: SUM(stock_quantity × unit_cost)
    Si unit_cost no disponible, usa unit_price como fallback.
    """

    def __init__(self):
        super().__init__()
        self.description = "Valor total del inventario a precio de coste"
        self.unit = "€"
        self.category = "Inventario"

    def calculate(self, context: QueryContext) -> float:
        """Calcular valor total del inventario."""
        # Calcular usando unit_cost, con fallback a unit_price
        result = context.inventory_query.with_entities(
            func.sum(
                InventorySnapshot.stock_quantity *
                func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
            ).label("stock_value")
        ).scalar()

        return float(result or 0)


class DeadStockValue(BaseMeasure):
    """
    Valor del stock sin movimiento > 180 días.

    Identifica productos con última venta hace más de 180 días.
    Fórmula: SUM(stock_quantity × unit_cost) WHERE days_since_sale > 180
    """

    def __init__(self):
        super().__init__()
        self.description = "Valor del stock sin movimiento en 180+ días"
        self.unit = "€"
        self.category = "Inventario"
        self.dead_stock_days = 180  # Configurable

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular valor del dead stock."""
        cutoff_date = date.today() - timedelta(days=self.dead_stock_days)

        # Filtrar productos sin venta reciente
        query = context.inventory_query.filter(
            (InventorySnapshot.last_sale_date < cutoff_date) |
            (InventorySnapshot.last_sale_date.is_(None))
        )

        result = query.with_entities(
            func.sum(
                InventorySnapshot.stock_quantity *
                func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
            ).label("dead_stock_value"),
            func.count(InventorySnapshot.id).label("dead_stock_products")
        ).first()

        dead_value = float(result.dead_stock_value or 0) if result else 0
        dead_products = int(result.dead_stock_products or 0) if result else 0

        return {
            "value": dead_value,
            "products_count": dead_products,
            "days_threshold": self.dead_stock_days,
        }


class StockCoverageDays(BaseMeasure):
    """
    Días de cobertura del stock actual.

    Fórmula: Stock actual / Venta diaria promedio (últimos 90 días)
    Interpretación: Días hasta agotamiento al ritmo actual de ventas.
    """

    def __init__(self):
        super().__init__()
        self.description = "Días de cobertura del stock actual"
        self.unit = "días"
        self.category = "Inventario"
        self.lookback_days = 90  # Período para calcular venta diaria

    def calculate(self, context: QueryContext) -> Union[float, Dict[str, Any]]:
        """Calcular días de cobertura del stock."""
        # 1. Obtener stock total actual (en unidades)
        stock_result = context.inventory_query.with_entities(
            func.sum(InventorySnapshot.stock_quantity).label("total_stock")
        ).scalar()

        total_stock = int(stock_result or 0)

        if total_stock == 0:
            return {
                "coverage_days": 0,
                "total_stock": 0,
                "daily_sales_avg": 0,
                "interpretation": "Sin stock",
                "alert_level": "danger",  # Issue #487: Consistencia con otros casos
            }

        # 2. Calcular venta diaria promedio (últimos N días)
        end_date = date.today()
        start_date = end_date - timedelta(days=self.lookback_days)

        sales_query = context.db.query(
            func.sum(SalesData.quantity).label("total_sold")
        ).filter(
            SalesData.pharmacy_id == context.filters.pharmacy_id,
            SalesData.sale_date >= start_date,
            SalesData.sale_date <= end_date,
        )

        # Apply product_codes filter if specified (Issue #470 code review fix)
        if context.filters.product_codes:
            sales_query = sales_query.filter(
                SalesData.codigo_nacional.in_(context.filters.product_codes)
            )

        sales_result = sales_query.scalar()

        total_sold = int(sales_result or 0)

        if total_sold == 0:
            return {
                "coverage_days": None,  # JSON-serializable (not float("inf"))
                "total_stock": total_stock,
                "daily_sales_avg": 0,
                "interpretation": "Sin ventas en el período - Cobertura indefinida",
                "alert_level": "secondary",  # Issue #487: Sin ventas = gris (info neutral)
            }

        daily_sales_avg = total_sold / self.lookback_days
        coverage_days = total_stock / daily_sales_avg

        # Interpretación según contexto farmacia española (Issue #487)
        # Las farmacias pueden pedir 2 veces/día al almacén
        # Retraso máximo típico: 1 semana
        if coverage_days < 7:
            interpretation = "Stock crítico - Riesgo de rotura"
            alert_level = "danger"
        elif coverage_days < 14:
            interpretation = "Stock bajo - Revisar pedido"
            alert_level = "warning"
        elif coverage_days < 30:
            interpretation = "Stock saludable"
            alert_level = "success"
        elif coverage_days < 60:
            interpretation = "Stock holgado"
            alert_level = "info"
        else:
            interpretation = "Sobrestock - Revisar rotación"
            alert_level = "secondary"

        return {
            "coverage_days": round(coverage_days, 1),
            "total_stock": total_stock,
            "daily_sales_avg": round(daily_sales_avg, 2),
            "lookback_days": self.lookback_days,
            "interpretation": interpretation,
            "alert_level": alert_level,  # Issue #487: Para colorear KPI en frontend
        }


class StockAlerts(BaseMeasure):
    """
    Lista de productos con problemas de stock.

    Tipos de alertas:
    - dead_stock: Sin ventas en >180 días
    - low_stock: Cobertura < 7 días (riesgo rotura)
    - overstock: Cobertura > 90 días (capital inmovilizado)
    - slow_rotation: Sin ventas recientes pero con stock alto
    """

    def __init__(self):
        super().__init__()
        self.description = "Productos con alertas de stock"
        self.unit = "alertas"
        self.category = "Inventario"
        # Umbrales configurables
        self.dead_stock_days = 180
        self.low_coverage_days = 7
        self.overstock_days = 90
        self.lookback_days = 90  # Para calcular ventas diarias

    def calculate(self, context: QueryContext) -> Dict[str, Any]:
        """Calcular alertas de stock por producto."""
        today = date.today()
        cutoff_dead = today - timedelta(days=self.dead_stock_days)
        lookback_start = today - timedelta(days=self.lookback_days)

        # 1. Obtener ventas por producto (últimos 90 días)
        sales_subq = (
            context.db.query(
                SalesData.codigo_nacional,
                func.sum(SalesData.quantity).label("units_sold"),
            )
            .filter(
                SalesData.pharmacy_id == context.filters.pharmacy_id,
                SalesData.sale_date >= lookback_start,
                SalesData.sale_date <= today,
                SalesData.codigo_nacional.isnot(None),
            )
            .group_by(SalesData.codigo_nacional)
            .subquery()
        )

        # 2. Obtener snapshot más reciente
        latest_date = (
            context.db.query(func.max(InventorySnapshot.snapshot_date))
            .filter(InventorySnapshot.pharmacy_id == context.filters.pharmacy_id)
            .scalar()
        )

        if not latest_date:
            return {
                "alerts": [],
                "summary": {
                    "dead_stock": 0,
                    "low_stock": 0,
                    "overstock": 0,
                    "total": 0,
                },
            }

        # 3. Consultar productos con stock y sus ventas
        inventory_query = (
            context.db.query(
                InventorySnapshot.product_code,
                InventorySnapshot.product_name,
                InventorySnapshot.stock_quantity,
                InventorySnapshot.last_sale_date,
                InventorySnapshot.product_type,
                (
                    InventorySnapshot.stock_quantity *
                    func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
                ).label("stock_value"),
                func.coalesce(sales_subq.c.units_sold, 0).label("units_sold"),
            )
            .outerjoin(
                sales_subq,
                InventorySnapshot.product_code == sales_subq.c.codigo_nacional
            )
            .filter(
                InventorySnapshot.pharmacy_id == context.filters.pharmacy_id,
                InventorySnapshot.snapshot_date == latest_date,
                InventorySnapshot.stock_quantity > 0,  # Solo productos con stock
            )
        )

        # Filtrar por product_type si se especifica
        # Issue #534: Incluir productos sin clasificar (product_type NULL)
        # para evitar mostrar 0 alertas cuando hay productos sin tipo asignado
        if context.filters.product_type:
            # Mapeo de tipos: "medicamento" -> "prescription", "venta_libre" -> "venta_libre"
            type_map = {
                "medicamento": "prescription",
                "prescription": "prescription",
                "venta_libre": "venta_libre",
                "otc": "venta_libre",
            }
            db_type = type_map.get(context.filters.product_type, context.filters.product_type)
            # Incluir productos del tipo especificado O sin clasificar
            inventory_query = inventory_query.filter(
                or_(
                    InventorySnapshot.product_type == db_type,
                    InventorySnapshot.product_type.is_(None),
                )
            )

        inventory_with_sales = inventory_query.all()

        # 4. Clasificar alertas
        alerts: List[Dict] = []
        summary = {"dead_stock": 0, "low_stock": 0, "overstock": 0}

        for row in inventory_with_sales:
            product_alerts = []
            stock_qty = int(row.stock_quantity or 0)
            stock_value = float(row.stock_value or 0)
            units_sold = int(row.units_sold or 0)
            last_sale = row.last_sale_date

            # Calcular cobertura en días
            daily_sales = units_sold / self.lookback_days if units_sold > 0 else 0
            coverage_days = stock_qty / daily_sales if daily_sales > 0 else None

            # A) Dead stock: sin ventas en 180+ días
            if last_sale and last_sale < cutoff_dead:
                days_since = (today - last_sale).days
                product_alerts.append({
                    "type": "dead_stock",
                    "severity": "danger",
                    "message": f"Sin ventas en {days_since} días",
                    "days": days_since,
                })
                summary["dead_stock"] += 1
            elif last_sale is None and units_sold == 0:
                # Sin fecha de venta y sin ventas recientes
                product_alerts.append({
                    "type": "dead_stock",
                    "severity": "danger",
                    "message": "Sin registro de ventas",
                    "days": None,
                })
                summary["dead_stock"] += 1

            # B) Low stock: cobertura < 7 días (solo si hay ventas)
            if coverage_days is not None and coverage_days < self.low_coverage_days:
                product_alerts.append({
                    "type": "low_stock",
                    "severity": "warning",
                    "message": f"Cobertura {coverage_days:.0f} días",
                    "coverage_days": round(coverage_days, 1),
                })
                summary["low_stock"] += 1

            # C) Overstock: cobertura > 90 días
            if coverage_days is not None and coverage_days > self.overstock_days:
                product_alerts.append({
                    "type": "overstock",
                    "severity": "info",
                    "message": f"Cobertura {coverage_days:.0f} días",
                    "coverage_days": round(coverage_days, 1),
                })
                summary["overstock"] += 1

            # Si tiene alguna alerta, añadir a la lista
            if product_alerts:
                alerts.append({
                    "product_code": row.product_code,
                    "product_name": row.product_name,
                    "stock_quantity": stock_qty,
                    "stock_value": round(stock_value, 2),
                    "coverage_days": round(coverage_days, 1) if coverage_days else None,
                    "last_sale_date": last_sale.isoformat() if last_sale else None,
                    "alerts": product_alerts,
                    "primary_alert": product_alerts[0]["type"],  # Para ordenar
                    "primary_severity": product_alerts[0]["severity"],
                })

        # 5. Ordenar: danger > warning > info, luego por valor de stock descendente
        severity_order = {"danger": 0, "warning": 1, "info": 2}
        alerts.sort(
            key=lambda x: (
                severity_order.get(x["primary_severity"], 3),
                -x["stock_value"]
            )
        )

        summary["total"] = len(alerts)

        # Issue #535: Calcular distribución de TODOS los productos por cobertura
        # Esto incluye productos sin alertas (saludables/holgados)
        coverage_distribution = self._calculate_coverage_distribution(
            inventory_with_sales,
        )

        return {
            "alerts": alerts[:50],  # Limitar a 50 alertas
            "summary": summary,
            "thresholds": {
                "dead_stock_days": self.dead_stock_days,
                "low_coverage_days": self.low_coverage_days,
                "overstock_days": self.overstock_days,
            },
            # Issue #535: Distribución completa para gráfico donut
            "coverage_distribution": coverage_distribution,
        }

    def _calculate_coverage_distribution(
        self, inventory_rows: List
    ) -> Dict[str, Any]:
        """
        Issue #535: Calcular distribución de TODOS los productos por bandas de cobertura.

        Bandas:
        - critical: <7 días
        - low: 7-14 días
        - healthy: 14-30 días
        - comfortable: 30-60 días
        - overstock: >60 días
        - unknown: Sin ventas (cobertura indefinida)

        Returns:
            Dict con counts por banda y totales
        """
        bands = {
            "critical": {"count": 0, "value": 0.0, "min": 0, "max": 7},
            "low": {"count": 0, "value": 0.0, "min": 7, "max": 14},
            "healthy": {"count": 0, "value": 0.0, "min": 14, "max": 30},
            "comfortable": {"count": 0, "value": 0.0, "min": 30, "max": 60},
            "overstock": {"count": 0, "value": 0.0, "min": 60, "max": float("inf")},
            "unknown": {"count": 0, "value": 0.0, "min": None, "max": None},
        }

        total_products = 0
        total_value = 0.0

        for row in inventory_rows:
            stock_qty = int(row.stock_quantity or 0)
            stock_value = float(row.stock_value or 0)
            units_sold = int(row.units_sold or 0)

            # Calcular cobertura en días
            daily_sales = units_sold / self.lookback_days if units_sold > 0 else 0
            coverage_days = stock_qty / daily_sales if daily_sales > 0 else None

            # Clasificar en banda
            if coverage_days is None:
                band = "unknown"
            elif coverage_days < 7:
                band = "critical"
            elif coverage_days < 14:
                band = "low"
            elif coverage_days < 30:
                band = "healthy"
            elif coverage_days < 60:
                band = "comfortable"
            else:
                band = "overstock"

            bands[band]["count"] += 1
            bands[band]["value"] += stock_value
            total_products += 1
            total_value += stock_value

        # Calcular porcentajes
        for band_id, band_data in bands.items():
            if total_products > 0:
                band_data["percentage"] = round(
                    band_data["count"] / total_products * 100, 1
                )
            else:
                band_data["percentage"] = 0.0
            band_data["value"] = round(band_data["value"], 2)

        return {
            "bands": bands,
            "total_products": total_products,
            "total_value": round(total_value, 2),
        }
